#!/usr/bin/env python
# Automatically generate XINDEX Exception file list based on CTest log files
# Please run CTest to generated the log files first before running this script
#
# To generate XINDEX Exception file list for routine(s) under package FOO
# python XindexExceptGen.py -l <logDir> -o <outputDir>
#                           -p Foo -r <routine1> <routine2> <routine3> ...
# where
#    logDir is the directory contains all XINDEX log files, normally under
#           <CMAKE_BUILD_DIR>/Testing/Temporary/
#    outputDir is the directory to stores all XINDEX Exception file list, normally
#           under <OSEHRA-Automated-Testing git repo>/XindexException/[Cache|GTM]
#
# To generate XINDEX Exception file list for packages Foo1 Foo2 Foo3
# python XindexExceptGen.py -l <logDir> -o <outputDir>
#                           -p Foo1 Foo2 Foo3
#
# Refer to  python XindexExceptGen.py -h for more information
#---------------------------------------------------------------------------
# Copyright 2011 The Open Source Electronic Health Record Agent
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#---------------------------------------------------------------------------
import re
import os.path
import argparse
import glob

# Regular expression to grep routines from the log file
Routine= re.compile('^(?P<routine>[A-Z|0-9][^ ]+) +\* \* .*[cC]hecksum:.*')
# regular expression to grep error message from the log file
ErrWarn=re.compile('W - |F -')

# Function to generate the XindexException File List
# @logFileName, the logFileName generated during CTest XINDEX run.
# @outputDir, the XindexException file outputDir
# @routineSet, the routineSet that need to be added, if empty, then all need to be added

def generateXindexExceptionList(logFileName, outputDir, routineSet=None):
    # Exception File List key file name, value is the file handler
    exceptionFileList=dict()
    isSetEmpty = routineSet == None or len(routineSet)== 0
    packageLogFile = open(logFileName,'r')
    packageName = os.path.basename(logFileName)
    # assume the package name is in the following format
    packageName=packageName[0:packageName.find("Test.log")]
    currentRoutineName=""
    for line in packageLogFile:
#        find all the routines names
        newRoutineName=Routine.search(line)
        if(newRoutineName):
            if (newRoutineName != currentRoutineName):
                currentRoutineName=newRoutineName.group('routine')
        else:
            ErrWarnLine =ErrWarn.search(line)
            if(ErrWarnLine and (isSetEmpty or currentRoutineName in routineSet)):
                # we found out the ErrWarnLine
                # check if the file is already exist and open
                outputFilename=packageName + "." + currentRoutineName
                if outputFilename in exceptionFileList:
                    fileHandle=exceptionFileList[outputFilename]
                else:
                    absPath = os.path.join(outputDir,outputFilename)
                    fileHandle=open(absPath,'wb')
                    exceptionFileList[outputFilename] = fileHandle
                fileHandle.write(line.strip()+"\n")
    packageLogFile.close()
    #close the file handle associated with the package
    while(len(exceptionFileList) > 0):
        (k,v)=exceptionFileList.popitem()
        v.close()

if __name__ == '__main__':
    parser = argparse.ArgumentParser(description='XIndex Exception List file generator')
    parser.add_argument('-l', required=True, dest='logDir',
                        help='Input XINDEX log files directory generated by CTest, nomally under'
                             'CMAKE_BUILD_DIR/Testing/Temporary/')
    parser.add_argument('-o', required=True, dest='outputDir',
                        help='Output directory to store the XINDEX Exception file list')

    parser.add_argument('-a',  dest="allPackage", action='store_true',
                        help='All packages')
    parser.add_argument('-p', dest='packageName', nargs='*',
                        help='only specified package names')
    parser.add_argument('-r', nargs='*', dest='routinesNames',
                         help='Specified routines under one package')
    result = vars(parser.parse_args())
    searchFiles=[]
    routines=set()
    if (result['allPackage']):
        logFilenamePattern="*Test.log"
        searchFiles = glob.glob(os.path.join(result['logDir'],logFilenamePattern))
    else:
        if (not result['packageName']):
            exit -1;
        for package in result['packageName']:
            searchFiles.append(os.path.join(result['logDir'],package.replace(' ', '_')+'Test.log'))
        if (len(result['packageName']) == 1):
            routines=result['routinesNames']

    for logFile in searchFiles:
        generateXindexExceptionList(logFile, result['outputDir'], routines)
