/*
 * Janus 4.0 (c)
 * Copyright (c) 2011 Hawaii Resource Group LLC. All Rights Reserved.
 * Developed for the Pacific Telehealth & Technology Hui and the Pacific Joint Information Technology Center
 * Contributors:
 *             Honorable Senator Daniel K. Inouye
 *             VA Pacific Islands Health Care System
 *             Tripler Army Medical Center
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with the License.
 *
 * You may obtain a copy of the License at:
 *
 *            http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and limitations under the License.
 */

/**
 * @class Ext.ux.Printer.BaseRenderer
 * @extends Object
 * @author Ed Spencer
 * Abstract base renderer class. Don't use this directly, use a subclass instead
 */
Ext.ux.Printer.BaseRenderer = Ext.extend(Object, {
  /**
   * Prints the component
   * @param {Ext.Component} component The component to print
   */
  print: function(component) {
  	var name = component && component.getXType
             ? String.format("print_{0}_{1}", component.getXType(), component.id.replace(/-/g, '_'))
             : "print";
             
    var win = window.open('', name);
    
    // gecko looses its document after document.close(). but fortunally waits with printing till css is loaded itself
    if (Ext.isGecko) {
      win.print();
      win.close();
      return;
    }
    
    win.document.write(this.generateHTML(component));
    win.document.close();

    this.doPrintOnStylesheetLoad.defer(10, this, [win]);
  },
  /**
   * Prints the component
   * @param {Ext.Component} component The component to print
   */
  printPatInfo: function(component) {
  	var name = component && component.getXType
             ? String.format("print_{0}_{1}", component.getXType(), component.id.replace(/-/g, '_'))
             : "print";

    var win = window.open('', name);

    // gecko looses its document after document.close(). but fortunally waits with printing till css is loaded itself
    if (Ext.isGecko) {
      win.print();
      win.close();
      return;
    }

    win.document.write(this.generatePatHTML(component));
    win.document.close();

    this.doPrintOnStylesheetLoad.defer(10, this, [win]);
  },
  /**
   * check if style is loaded and do print afterwards
   * 
   * @param {window} win
   */
  doPrintOnStylesheetLoad: function(win) {
    var el = win.document.getElementById('csscheck'),
        comp = el.currentStyle || getComputedStyle(el, null);
    if (comp.display !== "none") {
      this.doPrintOnStylesheetLoad.defer(10, this, [win]);
      return;
    }
    win.print();
    win.close();
  },
  
  /**
   * Generates the HTML Markup which wraps whatever this.generateBody produces
   * @param {Ext.Component} component The component to generate HTML for
   * @return {String} An HTML fragment to be placed inside the print window
   */
  generateHTML: function(component) {
    return new Ext.XTemplate(
      '<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML Transitional//EN"><META http-equiv="Content-Type" content="text/html; charset=utf-8">',
      '<HTML class="yui3-js-enabled  ext-border-box">',
        '<head>',
          '<meta content="text/html; charset=UTF-8" http-equiv="Content-Type" />',
          '<link href="' + this.stylesheetPath + '?' + new Date().getTime() + '" rel="stylesheet" type="text/css" media="screen,print" />',
          '<title>' + this.getTitle(component) + '</title>',
        '</head>',
        '<body>',
          '<div id="csscheck"></div>',
          this.generateBody(component),
        '</body>',
      '</html>'
    ).apply(this.prepareData(component));
  },

  /**
   * Generates the HTML Markup which wraps whatever this.generateBody produces
   * @param {Ext.Component} component The component to generate HTML for
   * @return {String} An HTML fragment to be placed inside the print window
   */
  generatePatHTML: function(component) {
    return new Ext.XTemplate(
      '<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML Transitional//EN"><META http-equiv="Content-Type" content="text/html; charset=utf-8">',
      '<HTML class="yui3-js-enabled  ext-border-box">',
        '<head>',
          '<meta content="text/html; charset=UTF-8" http-equiv="Content-Type" />',
          '<link href="' + this.stylesheetPath + '?' + new Date().getTime() + '" rel="stylesheet" type="text/css" media="screen,print" />',
          '<title>' + this.getTitle(component) + '</title>',
        '</head>',
        '<body>',
        ' <div id="csscheck"></div>',
          this.generateDemographics(),
          this.generateBody(component),
        '</body>',
      '</html>'
    ).apply(this.prepareData(component));
  },
  generateDemographics: function(){
    var htmlOut = '</div><div class="print_demo">' +
        '   <div id="demoPatName">Patient Name: <span class="print_demoValue">' + Ext.fly('demo_patName_' + Ext.getCmp('patDemographics').tabPanel.items.indexOf(Ext.getCmp('patDemographics').tabPanel.getActiveTab())).dom.innerText + '</span></div>' +
        '   <div id="demoPatDOB">DOB: <span class="print_demoValue">' + Ext.fly('demo_dob_' + Ext.getCmp('patDemographics').tabPanel.items.indexOf(Ext.getCmp('patDemographics').tabPanel.getActiveTab())).dom.innerText + '</span></div>' +
        '   <div id="demoPatGender">Gender: <span class="print_demoValue">' + Ext.fly('demo_gender_' + Ext.getCmp('patDemographics').tabPanel.items.indexOf(Ext.getCmp('patDemographics').tabPanel.getActiveTab())).dom.innerText + '</span></div>' +
        '   <div id="demoPatSSN">SSN: <span class="print_demoValue">' + Ext.fly('demo_ssn_' + Ext.getCmp('patDemographics').tabPanel.items.indexOf(Ext.getCmp('patDemographics').tabPanel.getActiveTab())).dom.innerText + '</span></div>' +
        '</div><div style="clear: both"></div>';
      return htmlOut;
  },
  /**
   * Returns the HTML that will be placed into the print window. This should produce HTML to go inside the
   * <body> element only, as <head> is generated in the print function
   * @param {Ext.Component} component The component to render
   * @return {String} The HTML fragment to place inside the print window's <body> element
   */
  generateBody: Ext.emptyFn,
  
  /**
   * Prepares data suitable for use in an XTemplate from the component 
   * @param {Ext.Component} component The component to acquire data from
   * @return {Array} An empty array (override this to prepare your own data)
   */
  prepareData: function(component) {
    return component;
  },
  
  /**
   * Returns the title to give to the print window
   * @param {Ext.Component} component The component to be printed
   * @return {String} The window title
   */
  getTitle: function(component) {
    return typeof component.getTitle == 'function' ? component.getTitle() : (component.title || "Printing");
  },
  
  /**
   * @property stylesheetPath
   * @type String
   * The path at which the print stylesheet can be found (defaults to 'stylesheets/print.css')
   */
  stylesheetPath: 'stylesheets/print.css'
});
