/*
 * Janus 4.0 (c)
 * Copyright (c) 2011 Hawaii Resource Group LLC. All Rights Reserved.
 * Developed for the Pacific Telehealth & Technology Hui and the Pacific Joint Information Technology Center
 * Contributors:
 *             Honorable Senator Daniel K. Inouye
 *             VA Pacific Islands Health Care System
 *             Tripler Army Medical Center
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with the License.
 *
 * You may obtain a copy of the License at:
 *
 *            http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and limitations under the License.
 */

/**
 * Provider Appointments Widget
 * 
 * @param {} config
 * 
 * @cfg provToken
 *     Provider's security token
 * @cfg desktop
 *    Reference to application desktop object
 */
Ext.hui.widget.provider.Appointments = function(config)
{
    var today =new Date();
    var ONE_DAY = 24*60*60*1000;

    //if(!config.endDay){
        this.endDate = new Date((today.getTime() + 14*ONE_DAY));
//    } else {
//        this.endDate = new Date((today.getTime() + config.endDay*ONE_DAY));
//    }
//    if(!config.startDay){
        this.startDate = new Date();
//    } else {
//        this.startDate = new Date((today.getTime() + config.startDay*ONE_DAY));
//    }

    Ext.hui.widget.provider.Appointments.superclass.constructor.call(this, config);
}

Ext.extend(Ext.hui.widget.provider.Appointments, Ext.hui.widget.Widget,
{
	title: "Appointments",
	height:200,
    layout:'border',
    tools:[
		{
			id: 'filter',
			qtip: 'Configure Filter',
            cls: 'x-tool-filter',
            overCls:'x-tool-filter-over',
			handler: function (event,toolEl, panel){
				if (panel.dateFilter) {
                    panel.dateFilter.toggleView();
				}
			}
		},
        {
			id: 'copy',
			qtip: 'Copy to Clipboard',
            cls: 'x-tool-copy',
            overCls:'x-tool-copy-over',
			handler: function (event,toolEl, panel){
				if (panel.gridPanel) {
                    Ext.hui.common.copyGridPanel(panel.gridPanel);
				}
			}
		},
		{
			id: 'print',
			qtip: 'Print',
			handler: function (event,toolEl, panel){
				Ext.ux.Printer.BaseRenderer.prototype.stylesheetPath = 'css/print.css';
				if (panel.gridPanel) {
					Ext.ux.Printer.printPatInfo(panel.gridPanel);
				}
			}
		},
		{
		    id:'refresh',
		    qtip: 'Refresh',
		    handler: function(event, toolEl, panel){
		        panel.dataStore.reload();
		    }
		},
		{
			id:'close',
		    qtip: 'Close Widget',
		    handler: function(event, toolEl, panel){
		        panel.destroy();
                Ext.hui.session.saveConfig();
		    }
		}
	],
	initComponent: function() {
		this.dataStore = new Ext.data.Store({
			restful:true,
			proxy: new Ext.data.HttpProxy({
						url: 'rest/providerAppointments.json',
						disableCaching:true,
						method:'GET',
                        timeout: Ext.hui.common.CONNECTION_TIMEOUT_MS
	    			}),
            baseParams:
            {
                patToken:this.patientToken,
                startDate:Ext.util.Format.dateRenderer("m/d/y")(this.startDate),
                endDate:Ext.util.Format.dateRenderer("m/d/y")(this.endDate)
            },
	        reader: this.createJsonReader(),
            sortInfo: {
                field: 'appointmentDate',
                direction: 'ASC' // or 'DESC' (case sensitive for local sorting)
            },

			listeners:
			{
				exception:{scope:this, fn:function(dataProxy, type, action, options, response, arg){
                    Ext.hui.common.handleNetworkFailure(response.status, this);

                }},
				beforeLoad:
                {
                    scope:this,
                    fn:function()
                    {
                        if (!this || this.isDestroy) return;
                        this.loading = new Ext.LoadMask(this.gridArea.el, {msg:"Loading...", store:this.dataStore});
                        this.loading.show();
                        if (this.dateFilter) this.dateFilter.disable();
                    }
                },
                load:
				{
					scope:this, 
					fn:
					  	function(store, records, options){
		    				if (!this || this.isDestroy) return;
                            var inMemoryStore = new Ext.data.Store({
                                proxy: new Ext.ux.data.PagingMemoryProxy(store.reader.jsonData),
                                remoteSort:true,
                                sortInfo: {
                                    field: 'appointmentDate',
                                    direction: 'ASC' // or 'DESC' (case sensitive for local sorting)
                                },
                                reader: this.createJsonReader()
                            });
                            //destroy previous grid panel
                            if (this.gridPanel)
                            {
                                this.gridPanel.destroy();
                            }
                            this.gridPanel = this.createGrid(inMemoryStore, this);
                            inMemoryStore.load({params:{start:0, limit:Ext.hui.common.WIDGET_NUM_RECORDS_PER_PAGE}});

                            this.gridArea.removeAll();
                            this.gridArea.add(this.gridPanel);
                            this.doLayout();
                            this.setTitle("Appointments ("+records.length+")");

                            if (this.dateFilter) this.dateFilter.enable();
	    				}
    			}
			}
	    });

        this.dateFilter = new Ext.hui.filter.DateFilter({
            region:'north',
            dataStore:this.dataStore,
            startDate:this.startDate,
            endDate:this.endDate,
            disabled:true,
            hidden:true
        });

        this.gridArea = new Ext.Container({layout:'fit', region:'center', html:''});
		this.items = [this.dateFilter, this.gridArea];

		Ext.hui.widget.provider.Appointments.superclass.initComponent.call(this);
	},
    createJsonReader: function()
    {
        return new Ext.data.JsonReader({
	            root: ''
	        },
			[
                {name: 'appointmentDate', type:'date'},
                {name: 'clinic'},
				{name: 'status'},
                {name: 'type'},
				{name: 'siteCode'},
                {name: 'providerName'},
                {name: 'reason'},
                {name: 'appointmentId'},
                {name: 'recordDetailsToken'},
                {name: 'patientToken'},
				{name: 'siteMoniker'},
                {name: 'patientName'},
                {name: 'patientToken'},
                {name: 'siteAgency'},
                {name: 'sourcePlatform'}
	        ]);
    },
    createGrid: function(gridDataStore, refWidget)
    {
        var dateRenderer = Ext.util.Format.dateRenderer('M d, Y H:i');

		return new Ext.grid.GridPanel({
	        store: gridDataStore,
	        columns: [
	            {
	                header   : 'Site',
	                width    : 40,
	                sortable : true,
	                dataIndex: 'siteMoniker',
	                renderer :
	                        function(value, metaData, record, rowIndex, colIndex, store) {
	                            return Ext.hui.common.formatSiteMonikerDisplay(record.data.siteAgency, record.data.siteMoniker);
	                        }
	            },
	            {
                    id: 'appointmentDate',
	                width    : 60,
	                header   : 'Date',
	                sortable : true,
	                dataIndex: 'appointmentDate',
	                renderer:{
                        fn:function(value, metaData, record, rowIndex, colIndex, store) {
                            if (record.data.recordDetailsToken != null)
                            {
                                return String.format('<span class="record-details" id="appointment-details-{0}-{1}">{2}</span>', refWidget.id, record.id, dateRenderer(record.data.appointmentDate));
                            }
                            else return dateRenderer(record.data.appointmentDate);

                        },
                        scope:this
                    }
	            },
                {
	                width    : 60,
	                header   : 'Clinic',
	                sortable : true,
	                dataIndex: 'clinic'
	            },
                {
                    id: 'patient',
	                header   : 'Patient',
	                width    : 80,
	                sortable : true,
	                dataIndex: 'patientName',
                    renderer :
	                    function(value, metaData, record, rowIndex, colIndex, store) {
	                        return String.format('<span id="patient-link-{0}-{1}" class="selectpatient" alt="Link Select Patient">'+record.data.patientName+'</span>', refWidget.id, record.id);
	                    }

	            }
	        ],
	        listeners:{
	        	rowclick:
	        	{
	        		scope:this,
	        		fn:function(grid, rowIdx, evtObj) {
	        			var record = grid.getStore().getAt(rowIdx);
                        if(evtObj.within(String.format("patient-link-{0}-{1}", refWidget.id, record.id)))
                        {
                            Ext.hui.app.selectPatient(record.data.patientToken, record.data.patientName, refWidget.id);
                        }
                        if (Ext.get(String.format("appointment-details-{0}-{1}", refWidget.id, record.id))){
                            if(evtObj.within(String.format("appointment-details-{0}-{1}", refWidget.id, record.id)))
                            {
                                refWidget.desktop.createWindow({
                                    title:String.format("Appointment Details ~ {0} {1} :: {2}",
                                        Ext.hui.common.formatSiteMonikerDisplay(record.data.siteAgency, record.data.siteMoniker),
                                        record.data.clinic, record.data.patientName),
                                    patientToken:record.data.patientToken,
                                    desktop:refWidget.desktop,
                                    recordDetailsToken:record.data.recordDetailsToken
                                }, Ext.hui.desktopwindow.patient.AppointmentDetails).show();
                             }
                        }
	        		}
	        	}
	        },

            selModel: new Ext.grid.CellSelectionModel(),
	        stripeRows: true,
            columnLines:true,
	        autoExpandColumn:'patient',
	        border:true,
	        bodyBorder:true,
	        loadMask:true,
            layout:'fit',
            region: 'center',
            bbar: new Ext.PagingToolbar({
                pageSize: Ext.hui.common.WIDGET_NUM_RECORDS_PER_PAGE,
                store: gridDataStore,
                displayInfo: true,
                items:[
                {
                    text: 'more >>',
                    cls: 'more-link',
                    listeners:
			        {
                        click: function(btn, pressed){
                            refWidget.desktop.createWindow({
                                desktop:refWidget.desktop,
                                startDate:refWidget.startDate,
                                endDate:refWidget.endDate
                            }, Ext.hui.desktopwindow.provider.Appointments).show();
                        }

                    }
                }]
            }),
	        viewConfig: {
	        	emptyText:"No results",
				forceFit:true
	        }
	    });
    }
});

Ext.reg('prov_appointments_widget', Ext.hui.widget.provider.Appointments);