/*
 * Janus 4.0 (c)
 * Copyright (c) 2011 Hawaii Resource Group LLC. All Rights Reserved.
 * Developed for the Pacific Telehealth & Technology Hui and the Pacific Joint Information Technology Center
 * Contributors:
 *             Honorable Senator Daniel K. Inouye
 *             VA Pacific Islands Health Care System
 *             Tripler Army Medical Center
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with the License.
 *
 * You may obtain a copy of the License at:
 *
 *            http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and limitations under the License.
 */

/**
 * Vitals Widget
 * 
 * @param {} config
 * 
 * @cfg patientToken
 *     Patient's security token
 * @cfg desktop
 *    Reference to application desktop object
 */
Ext.hui.widget.patient.Vitals = function(config)
{
    var today =new Date();
    var ONE_DAY = 24*60*60*1000;

//    if(!config.endDay){
        this.endDate = today;
//    } else {
//        this.endDate = new Date((today.getTime() + config.endDay*ONE_DAY));
//    }
//    if(!config.startDay){
        this.startDate = new Date((today.getTime() - 365*ONE_DAY));
//    } else {
//        this.startDate = new Date((today.getTime() + config.startDay*ONE_DAY));
//    }
    if(Ext.hui.app.EXTEND_YEAR=="true"){
                this.startDate.setYear(Ext.hui.app.EXTEND_YEAR_DATE);
            }

    Ext.hui.widget.patient.Vitals.superclass.constructor.call(this, config);
}

Ext.extend(Ext.hui.widget.patient.Vitals, Ext.hui.widget.Widget, 
{
    title: "Vitals",
    height:200,
    layout:'border',
    tools:[
        {
			id: 'filter',
			qtip: 'Configure Filter',
            cls: 'x-tool-filter',
            overCls:'x-tool-filter-over',
			handler: function (event,toolEl, panel){
				if (panel.filter) {
                    panel.filter.toggleView();
				}
			}
		},
        {
            id:'help',
            qtip: 'Interface Status',
            handler: function(event, toolEl, panel)
            {
                var win = panel.desktop.createWindow({
                    title:"Interface Status ~ Vitals",
                    width: 600,
                    height: 500,
                    autoScroll: true
                });
                win.show();
                var sourceStatus = Ext.hui.app.BHIE_STATUS_NO_REPORT_MSG;

                if (panel.bhieSourceReport != null)
                {
                    sourceStatus = panel.bhieSourceReport;
                }

                win.body.dom.innerHTML = "<div class='bhie-report'>" + sourceStatus + "</div>";
            }
        },
        {
			id: 'copy',
			qtip: 'Copy to Clipboard',
            cls: 'x-tool-copy',
            overCls:'x-tool-copy-over',
			handler: function (event,toolEl, panel){
				if (panel.gridPanel) {
                    Ext.hui.common.copyGridPanel(panel.gridPanel);
				}
			}
		},
		{
			id: 'print',
			qtip: 'Print',
			handler: function (event,toolEl, panel){
				Ext.ux.Printer.BaseRenderer.prototype.stylesheetPath = 'css/print.css';
				if (panel.gridPanel) {
					Ext.ux.Printer.printPatInfo(panel.gridPanel);
				}
			}
		},
        {
            id:'refresh',
            qtip: 'Refresh',
            handler: function(event, toolEl, panel){
                panel.dataStore.reload();
            }
        },
        {
            id:'close',
            qtip: 'Close Widget',
            handler: function(event, toolEl, panel){
                panel.destroy();
            }
        }
    ],
    initComponent: function() {

        this.bhieSourceReport = null;

        this.dataStore = new Ext.data.GroupingStore({
            restful:true,
            proxy: new Ext.data.HttpProxy({
                url: 'rest/patientVitals.json',
                disableCaching:true,
                timeout: Ext.hui.common.CONNECTION_TIMEOUT_MS,
                method:'GET'
            }),
            baseParams:{
                patToken:this.patientToken,
                startDate:Ext.util.Format.dateRenderer("m/d/y")(this.startDate),
                endDate:Ext.util.Format.dateRenderer("m/d/y")(this.endDate)
            },
            reader: this.createJsonReader(),
            groupField:'dateTaken',
            groupDir:'DESC',
            sortInfo: {field: 'dateTaken', direction: 'DESC'},
            listeners:
            {
                exception:{
                    scope:this, 
                    fn:function(dataProxy, type, action, options, response, arg){
                        Ext.hui.common.handleNetworkFailure(response.status, this);

                    }
                },
                beforeLoad:
                {
                    scope:this,
                    fn:function()
                    {
                        if (!this || this.isDestroy) return;
                        this.loading = new Ext.LoadMask(this.gridArea.el, {msg:"Loading...", store:this.dataStore});
                        this.loading.show();
                        if (this.filter) this.filter.disable();
                        this.bhieSourceReport = null;
                    }
                },
                load:
                {
                    scope:this,
                    fn:
                    function(store, records, options){
                        if (!this || this.isDestroy) return;

                        var jsonDt = [];
                        var bhieReportRecIdx = -1;
                        for(var i = 0; i < records.length; i++)
                        {
                            jsonDt[i] = records[i].json;
                            if (jsonDt[i].siteMoniker == Ext.hui.app.BHIE_STATUS_REPORT_FLAG)
                            {
                                this.bhieSourceReport = jsonDt[i].type;
                                bhieReportRecIdx = i;
                            }
                        }

                        //remove bhie report from list
                        if (bhieReportRecIdx > -1)
                        {
                           jsonDt.remove(jsonDt[bhieReportRecIdx]);
                           store.remove(records[bhieReportRecIdx]);
                        }

                        var inMemoryStore = new Ext.data.GroupingStore({
                            proxy: new Ext.ux.data.PagingMemoryProxy(jsonDt),
                            remoteSort:true,
                            groupField:'dateTaken',
                            groupDir:'DESC',
                            reader: this.createJsonReader(),
                            sortInfo: {field: 'dateTaken', direction: 'DESC'}
                        });
                        //destroy previous grid panel
                        if (this.gridPanel)
                        {
                            this.gridPanel.destroy();
                        }
                        this.gridPanel = this.createGrid(inMemoryStore, this);
                        inMemoryStore.load({params:{start:0, limit:Ext.hui.common.WIDGET_NUM_RECORDS_PER_PAGE}});

                        this.gridArea.removeAll();
                        this.gridArea.add(this.gridPanel);
                        this.gridArea.doLayout();

                        this.setTitle("Vitals ("+jsonDt.length+")");

                        if (this.filter) this.filter.enable();
                    }
                }

            }
        });

        this.filter = new Ext.hui.filter.DateFilter({
            region:'north',
            dataStore:this.dataStore,
            startDate:this.startDate,
            endDate:this.endDate,
            disabled:true,
            hidden:true
        });

        this.gridArea = new Ext.Container({layout:'fit', region:'center', html:''});
        this.items = [this.filter, this.gridArea];

        Ext.hui.widget.patient.Vitals.superclass.initComponent.call(this);
    },
    createJsonReader: function()
    {
        return new Ext.data.JsonReader({
                root: ''
            },
            [
                {name: 'type'},
                {name: 'dateTaken', type:'date'},
                {name: 'result'},
                {name: 'units'},
                {name: 'qualifiers'},
                {name: 'histogramRecordToken'},
                {name: 'siteCode'},
                {name: 'siteMoniker'},
                {name: 'siteAgency'},
                {name: 'sourcePlatform'}
            ]);
    },
    createGrid: function(gridDataStore, refWidget)
    {
        var pagingBarItems = [
        {
            text: 'more >>',
            cls: 'more-link',
            listeners:
            {
                click: function(btn, pressed){
                    refWidget.desktop.createWindow({
                        patientToken:refWidget.patientToken,
                        desktop:refWidget.desktop,
                        startDate:refWidget.startDate,
                        endDate:refWidget.endDate
                    }, Ext.hui.desktopwindow.patient.Vitals).show();
                }

            }
        }];

        var dateRenderer = Ext.util.Format.dateRenderer('M d, Y H:i');

        return new Ext.grid.GridPanel({
            store: gridDataStore,
            columns: [
            {
                header   : 'Site',
                width    : 120,
                dataIndex: 'siteMoniker',
                renderer :
                function(value, metaData, record, rowIndex, colIndex, store) {
                    return Ext.hui.common.formatSiteMonikerDisplay(record.data.siteAgency, record.data.siteMoniker);
                }
            },

            {
                width    : 125,
                header   : 'Type',
                sortable : true,
                dataIndex: 'type'
            },
            {
                width    : 80,
                header   : 'Histogram',
                renderer :{
                    fn:function(value, metaData, record, rowIndex, colIndex, store) {
                        if (Ext.hui.common.isVitalGraphAllowed(record.data.type))
                        {
                            return String.format('<span class="histogram-link" id="histogram-graph-link-{0}-{1}">graph</span>&nbsp;|&nbsp;<span class="histogram-link" id="histogram-table-link-{0}-{1}">table</span>', refWidget.id, record.id);
                        }
                        else return "Unavailable";
                    },
                    scope:this
                }
            },
            {
                width    : 80,
                header   : 'Result',
                sortable : true,
                dataIndex: 'result'
            },
            {
                width    : 80,
                header   : 'Units',
                sortable : true,
                dataIndex: 'units'
            },
            {
                width    : 125,
                header   : 'Date Taken',
                sortable : true,
                dataIndex: 'dateTaken',
                renderer: dateRenderer
            }
            ],
            listeners:{
                rowclick:
                {
                    scope:this,
                    fn:function(grid, rowIdx, evtObj) {
                        var record = grid.getStore().getAt(rowIdx);
                        if (Ext.get(String.format("histogram-details-{0}-{1}", refWidget.id, record.id))
                                || Ext.get(String.format("histogram-table-link-{0}-{1}", refWidget.id, record.id))){
                            if(evtObj.within(String.format("histogram-graph-link-{0}-{1}", refWidget.id, record.id)))
                            {
                                refWidget.desktop.createWindow({
                                    title:String.format("Vitals Graph ~ {0}",
                                        record.data.type),
                                        histogramRecordToken:record.data.histogramRecordToken,
                                        patientToken:refWidget.patientToken
                                }, Ext.hui.desktopwindow.patient.VitalsGraph).show();
                            }
                            else if(evtObj.within(String.format("histogram-table-link-{0}-{1}", refWidget.id, record.id)))
                            {

                                refWidget.desktop.createWindow({
                                     title:String.format("Vitals Table ~ {0}",
                                         record.data.type),
                                    histogramRecordToken:record.data.histogramRecordToken,
                                    patientToken:refWidget.patientToken,
                                    desktop:refWidget.desktop
                                }, Ext.hui.desktopwindow.patient.VitalsTable).show();
                            }
                        }
                    }
                }
            },
            view: new Ext.grid.GroupingView({
                       forceFit:true,
                       groupTextTpl: '{text}'
                   }),
            selModel: new Ext.grid.CellSelectionModel(),
            stripeRows: true,
            border:true,
            bodyBorder:true,
            columnLines:true,
            loadMask:true,
            layout:'fit',
            region: 'center',
            bbar: new Ext.PagingToolbar({
                pageSize: Ext.hui.common.WIDGET_NUM_RECORDS_PER_PAGE,
                store: gridDataStore,
                displayInfo: true,
                items:pagingBarItems
            }),
            viewConfig: {
                emptyText:"No results",
                forceFit:true
            }
        });
    }
});

Ext.reg('vitals_widget', Ext.hui.widget.patient.Vitals);