/*
 * Janus 4.0 (c)
 * Copyright (c) 2011 Hawaii Resource Group LLC. All Rights Reserved.
 * Developed for the Pacific Telehealth & Technology Hui and the Pacific Joint Information Technology Center
 * Contributors:
 *             Honorable Senator Daniel K. Inouye
 *             VA Pacific Islands Health Care System
 *             Tripler Army Medical Center
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with the License.
 *
 * You may obtain a copy of the License at:
 *
 *            http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and limitations under the License.
 */

/**
 * Radiology Exams Widget
 *
 * @param {} config
 *
 * @cfg patientToken
 *     Patient's security token
 * @cfg desktop
 *    Reference to application desktop object
 */
Ext.hui.widget.patient.RadiologyExams = function(config)
{
    var today =new Date();
    var ONE_DAY = 24*60*60*1000;

//    if(!config.endDay){
        this.endDate = today;
//    } else {
//        this.endDate = new Date((today.getTime() + config.endDay*ONE_DAY));
//    }
//    if(!config.startDay){
        this.startDate = new Date((today.getTime() - 365*ONE_DAY));
//    } else {
//        this.startDate = new Date((today.getTime() + config.startDay*ONE_DAY));
//    }
    if(Ext.hui.app.EXTEND_YEAR=="true"){
                this.startDate.setYear(Ext.hui.app.EXTEND_YEAR_DATE);
            }

    Ext.hui.widget.patient.RadiologyExams.superclass.constructor.call(this, config);
}

Ext.extend(Ext.hui.widget.patient.RadiologyExams, Ext.hui.widget.Widget,
{
    title: "Radiology Exams",
    height:200,
    layout:'border',
    tools:[
        {
			id: 'filter',
			qtip: 'Configure Filter',
            cls: 'x-tool-filter',
            overCls:'x-tool-filter-over',
			handler: function (event,toolEl, panel){
				if (panel.dateFilter) {
                    panel.dateFilter.toggleView();
				}
			}
		},
        {
			id: 'down',
			qtip: 'Download and install Medweb plugin',
			handler: function (event,toolEl, panel){
                window.location.href = Ext.hui.app.MEDWEB_PLUGIN_URL;
            }
		},
        {
			id: 'copy',
			qtip: 'Copy to Clipboard',
            cls: 'x-tool-copy',
            overCls:'x-tool-copy-over',
			handler: function (event,toolEl, panel){
				if (panel.gridPanel) {
                    Ext.hui.common.copyGridPanel(panel.gridPanel);
				}
			}
		},
		{
			id: 'print',
			qtip: 'Print',
			handler: function (event,toolEl, panel){
				Ext.ux.Printer.BaseRenderer.prototype.stylesheetPath = 'css/print.css';
				if (panel.gridPanel) {
					Ext.ux.Printer.printPatInfo(panel.gridPanel);
				}
			}
		},
        {
            id:'refresh',
            qtip: 'Refresh',
            handler: function(event, toolEl, panel){
                panel.dataStore.reload();
            }
        },
        {
            id:'close',
            qtip: 'Close Widget',
            handler: function(event, toolEl, panel){
                panel.destroy();
                Ext.hui.session.saveConfig();
            }
        }
    ],
    initComponent: function() {

        this.dataStore = new Ext.data.Store({
            restful:true,
            proxy: new Ext.data.HttpProxy({
                url: 'rest/patientRadiologyExams.json',
                disableCaching:true,
                timeout: Ext.hui.common.CONNECTION_TIMEOUT_MS,
                method:'GET'
            }),
            baseParams:{
                patToken:this.patientToken,
                startDate:Ext.util.Format.dateRenderer("m/d/y")(this.startDate),
                endDate:Ext.util.Format.dateRenderer("m/d/y")(this.endDate)
            },
            reader: this.createJsonReader(),

            sortInfo: {
                field: 'examDate',
                direction: 'DESC'
            },
            listeners:
            {
                exception:{
                    scope:this,
                    fn:function(dataProxy, type, action, options, response, arg){
                        Ext.hui.common.handleNetworkFailure(response.status, this);

                    }
                },
                beforeLoad:
                {
                    scope:this,
                    fn:function()
                    {
                        if (!this || this.isDestroy) return;
                        this.loading = new Ext.LoadMask(this.gridArea.el, {msg:"Loading...", store:this.dataStore});
                        this.loading.show();
                        if (this.dateFilter) this.dateFilter.disable();
                    }
                },
                load:
                {
                    scope:this,
                    fn:
                    function(store, records, options){
                        if (!this || this.isDestroy) return;
                        var inMemoryStore = new Ext.data.Store({
                            proxy: new Ext.ux.data.PagingMemoryProxy(store.reader.jsonData),
                            remoteSort:true,
                            reader: this.createJsonReader(),
                            sortInfo: {
                                field: 'examDate',
                                direction: 'DESC'
                            }
                        });
                        //destroy previous grid panel
                        if (this.gridPanel)
                        {
                            this.gridPanel.destroy();
                        }
                        this.gridPanel = this.createGrid(inMemoryStore, this);
                        inMemoryStore.load({params:{start:0, limit:Ext.hui.common.WIDGET_NUM_RECORDS_PER_PAGE}});

                        this.gridArea.removeAll();
                        this.gridArea.add(this.gridPanel);
                        this.doLayout();

                        this.setTitle("Radiology Exams ("+records.length+")");

                        if (this.dateFilter) this.dateFilter.enable();
                    }
                }

            }
        });

        this.dateFilter = new Ext.hui.filter.DateFilter({
            region:'north',
            dataStore:this.dataStore,
            startDate:this.startDate,
            endDate:this.endDate,
            disabled:true,
            hidden:true
        });

        this.gridArea = new Ext.Container({layout:'fit', region:'center', html:''});
        this.items = [this.dateFilter, this.gridArea];

        Ext.hui.widget.patient.RadiologyExams.superclass.initComponent.call(this);
    },
    createJsonReader: function()
    {
        return new Ext.data.JsonReader({
                root: ''
            },
            [
                {name: 'study'},
                {name: 'examDate', type:'date'},
                {name: 'recordDetailsToken'},
                {name: 'medWebAvailable'},
                {name: 'medWebToken'},
                {name: 'siteCode'},
                {name: 'siteMoniker'},
                {name: 'siteAgency'},
                {name: 'sourcePlatform'}
            ]);
    },
    createGrid: function(gridDataStore, refWidget)
    {
        var dateRenderer = Ext.util.Format.dateRenderer('M d, Y H:i');

        return new Ext.grid.GridPanel({
            store: gridDataStore,
            columns: [
                {
                    header   : 'Site',
                    width    : 60,
                    sortable : true,
                    dataIndex: 'siteMoniker',
                    renderer :
                    function(value, metaData, record, rowIndex, colIndex, store) {
                        return Ext.hui.common.formatSiteMonikerDisplay(record.data.siteAgency, record.data.siteMoniker);
                    }
                },
                {
                    width    : 100,
                    header   : 'Date',
                    sortable : true,
                    dataIndex: 'examDate',
                    renderer: dateRenderer
                },
                {
                    id: 'study',
                    width    : 100,
                    header   : 'Study',
                    sortable : true,
                    dataIndex: 'study',
                    renderer :{
                        fn:function(value, metaData, record, rowIndex, colIndex, store) {
                            if (record.data.recordDetailsToken != null)
                            {
                                return String.format('<span class="record-details" id="radiologyreport-details-{0}-{1}">{2}</span>', refWidget.id, record.id, record.data.study);
                            }
                            else return record.data.study;

                        },
                        scope:this
                    }
                },
                {
                    width:40,
                    header: 'Image',
                    id:'image',
                    sortable: 'false',
                    renderer:{
                        fn:function(value, metaData, record, rowIndex, colIndex, store) {
                            if (record.data.recordDetailsToken != null)
                            {
                                if(!record.data.medWebAvailable)
                                {
                                    return "";
                                }

                                var medWebURL = String.format("patientMedWeb?patToken={0}&recToken={1}",
                                        refWidget.patientToken, record.data.medWebToken);

                                var medWebCmd = String.format("Ext.hui.common.openAuxBrowserWindow(\"{0}\", \"IEHR_MEDWEB\");return false;", medWebURL);

                                return String.format('<span class="medweb-link" title="View in MedWeb" onclick=\'{0}\' alt="Link View in Medweb"><img src="images/camera_icon.png"  alt=\'Camera Icon\'/></span>', medWebCmd);
                            }
                            return "";

                        },
                        scope:this
                    }
                }
            ],
            listeners:{
                rowclick:
                {
                    scope:this,
                    fn:function(grid, rowIdx, evtObj) {
                        var record = grid.getStore().getAt(rowIdx);
                        if (Ext.get(String.format("radiologyreport-details-{0}-{1}", refWidget.id, record.id))){
                        if(evtObj.within(String.format("radiologyreport-details-{0}-{1}", refWidget.id, record.id)))
                        {
                            refWidget.desktop.createWindow({
                                title:String.format("Radiology Report ~ {0} {1}",
                                        Ext.hui.common.formatSiteMonikerDisplay(record.data.siteAgency, record.data.siteMoniker),
                                        record.data.study),
                                patientToken:refWidget.patientToken,
                                desktop:refWidget.desktop,
                                recordDetailsToken:record.data.recordDetailsToken
                            }, Ext.hui.desktopwindow.patient.RadiologyReport).show();
                        }
                        }
                    }
                }
            },
            selModel: new Ext.grid.CellSelectionModel(),
            stripeRows: true,
            autoExpandColumn:'study',
            border:true,
            bodyBorder:true,
            columnLines:true,
            loadMask:true,
            layout:'fit',
            region: 'center',
            bbar: new Ext.PagingToolbar({
                pageSize: Ext.hui.common.WIDGET_NUM_RECORDS_PER_PAGE,
                store: gridDataStore,
                displayInfo: true,
                items:[
                {
                    text: 'more >>',
                    cls: 'more-link',
                    listeners:
			        {
                        click: function(btn, pressed){
                            refWidget.desktop.createWindow({
                                patientToken:refWidget.patientToken,
                                desktop:refWidget.desktop,
                                startDate:refWidget.startDate,
                                endDate:refWidget.endDate
                            }, Ext.hui.desktopwindow.patient.RadiologyExams).show();
                        }

                    }
                }]
            }),
            viewConfig: {
                emptyText:"No results",
                forceFit:true
            }
        });
    }
});

Ext.reg('radiologyexams_widget', Ext.hui.widget.patient.RadiologyExams);