/*
 * Janus 4.0 (c)
 * Copyright (c) 2011 Hawaii Resource Group LLC. All Rights Reserved.
 * Developed for the Pacific Telehealth & Technology Hui and the Pacific Joint Information Technology Center
 * Contributors:
 *             Honorable Senator Daniel K. Inouye
 *             VA Pacific Islands Health Care System
 *             Tripler Army Medical Center
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with the License.
 *
 * You may obtain a copy of the License at:
 *
 *            http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and limitations under the License.
 */

/**
 * Patient Demographics Widget
 * 
 * @param {} config
 * 
 * @cfg patientToken
 *     Patient's security token
 * @cfg desktop
 *    Reference to application desktop object
 */
Ext.hui.widget.patient.PatientDemographics = function(config)
{
	this.patientToken = config.patientToken;
	this.desktop = config.desktop;
	
	Ext.hui.widget.patient.PatientDemographics.superclass.constructor.call(this, config);
	
	this.tabPanel = new Ext.TabPanel({
		activeTab: 0,
		bodyBorder:false,
        border:false,
        tabPosition:'top',
        enableTabScroll:true,
        plain:true,
        boxMinWidth: 275,
        style: 'z-index:999'
	});

	this.add(this.tabPanel);
	this.doLayout();

	var updater = this.tabPanel.getUpdater();
	updater.setRenderer(new Ext.hui.widget.patient.DemographicsRenderer(this));

	this.tabPanel.load({
		url: 'rest/patientDemographics.json',
		method:'GET',
		params: {patToken:this.patientToken},
		discardUrl: true,
	    nocache: true,
	    text: 'Loading...',
	    timeout:Ext.hui.common.CONNECTION_TIMEOUT_MS,
	    scripts: false,
	    scope:this,
	    callback:function(options, isSuccess, response)
		{
			if (!isSuccess)
			{
				Ext.hui.common.handleNetworkFailure(response.status, this);
			}
		}
	});

}

Ext.extend(Ext.hui.widget.patient.PatientDemographics, Ext.hui.widget.Widget, 
{	
	title: "Patient Demographics",
	header:false,
	bodyBorder:false,
	border:false,
	cls:"demographics",
	draggable:false

});

Ext.reg('patientdemographics_widget', Ext.hui.widget.patient.PatientDemographics);

/**
 * Demographics Renderer, parses json response and handles layout.
 */
Ext.hui.widget.patient.DemographicsRenderer = function(patDemoWidget)
{
	this.patDemoWidget = patDemoWidget;
	Ext.hui.widget.patient.DemographicsRenderer.superclass.constructor.call(this); 
}

Ext.extend(Ext.hui.widget.patient.DemographicsRenderer, Ext.Updater.BasicRenderer,
{
	render: function(el, xhr, updateManager, callback )
	{
		
		var reader = 
			new Ext.data.JsonReader({
	            root: ''
	        }, 
	        [
				{name: 'name'},
				{name: 'age'},
				{name: 'gender'},
				{name: 'fmpssn'},
				{name: 'dob'},
				{name: 'address1'},
				{name: 'address2'},
				{name: 'city'},
				{name: 'state'},
				{name: 'zipcode'},
				{name: 'fmpssn'},
				{name: 'percentServiceConnected'},
				{name: 'phone1'},
				{name: 'phone2'},
				{name: 'age'},
				{name: 'dob'},
				{name: 'gender'},
				{name: 'pcm'},
				{name: 'pcmPhone'},
				{name: 'clinic'},
				{name: 'codeGray'},
				{name: 'codeGrayMessage'},
                {name: 'siteName'},
				{name: 'siteCode'},
				{name: 'siteMoniker'},
                {name: 'siteAgency'},
                {name: 'sourcePlatform'}
	        ]);
		var jsonResponse = reader.read(xhr);
		el.dom.innerHTML = "";
		
		var records = jsonResponse.records;
		var codeGrayAlerts = [];
		if (this.patDemoWidget.tabPanel)
	    {
			for(var i = 0; i < records.length; i++)
			{
				var tpl = new Ext.XTemplate
				( 
					'<div id="demo_wrapper_'+ i +'" class="demo_wrapper">',
					'	<div id="demo_patName_'+ i +'" class="demo_patName">',
					'		<span id="demographic_name_{panelId}">{name}</span> {[Ext.hui.common.formatSiteIconDisplay(values.siteAgency)]} ',
					'		<tpl if="codeGray">',
					'			&nbsp;<span id="demographic_alerts_{panelId}" class="codegray">Alerts</span>&nbsp;',
					'		</tpl>', 
					'	</div>',
					'	<div id="demo_infoCol1_'+ i +'" class="demo_infoCol1">',
					'		<table>' +
                    '           <tr><td>{address1}</td></tr>' +
                    '           <tr><td>{address12}</td></tr>' +
                    '           <tr><td>' +
					'		    	{city}',
					'		    	<tpl if="this.isCity(city)">',
					'		    		, ',
					'		    	</tpl>',
					'		    	{state} {zipcode}',
                    '           </td></tr>' +
                    '       </table>',
					'	</div>',
					'	<div id="demo_infoCol2_'+ i +'" class="demo_infoCol2">',
					'		<div><span class="demo_cat">SSN: </span><span id="demo_ssn_'+ i +'">{fmpssn}</span></div>',
					'		<div><span class="demo_cat">DOB: </span><span id="demo_dob_'+ i +'">{dob}</span></div>',
					'		<div><span class="demo_cat">(h): </span><span id="demo_phone1_'+ i +'">{phone1}</span></div>',
					'		<div><span class="demo_cat">(w): </span><span id="demo_phone2_'+ i +'">{phone2}</span></div>',
					'		<div><span class="demo_cat">Gender: </span><span id="demo_gender_'+ i +'">{gender}</span></div>',
					'	</div><div style="clear:both"></div>',
					'</div>',
					{
						isCity: function(city)
						{
							return city.length > 0;
						}
					}
				);
				
				var data = records[i].data;
				var panel = new Ext.Panel({
						title:Ext.hui.common.formatSiteMonikerDisplay(data.siteAgency, data.siteMoniker),
                        tabTip:data.siteName,
						cls:"demographics-tab",
						bodyBorder:false,
        				border:false
                });
                if (data.codeGray)
                {
                    var codeGrayAlert = {};
                    codeGrayAlert.title = Ext.hui.common.formatSiteMonikerDisplay(data.siteAgency, data.siteMoniker) + " Alerts";
                    codeGrayAlert.msg = data.codeGrayMessage;
                    codeGrayAlerts.push(codeGrayAlert);
                }
				panel.on("afterrender", 
					function() {
								
						var showDemographicDetails = function()
						{
							this.widget.desktop.createWindow(
								{patientToken:this.widget.patientToken,
                                    desktop:this.widget.desktop//,
                                //    cls:'x-portlet'
                                },
								Ext.hui.desktopwindow.patient.PatientDemographicDetails).show();
						}
						
						if (this.demographicsData.codeGray)
						{
							Ext.EventManager.addListener('demographic_alerts_'+this.panel.id, 'click',
                                    function(){
                                        Ext.Msg.alert(
                                                Ext.hui.common.formatSiteMonikerDisplay(this.demographicsData.siteAgency, this.demographicsData.siteMoniker) + " Alerts",
                                                this.demographicsData.codeGrayMessage
                                        );
                                    },
                                    this
                            );
						}
						
						Ext.EventManager.addListener('demographic_name_'+this.panel.id, 'click', showDemographicDetails, this);
					}, {widget:this.patDemoWidget, panel:panel, demographicsData:data});	

				data.panelId = panel.id;
				var html = tpl.apply(data);
				panel.html = html;
				
				this.patDemoWidget.tabPanel.add(panel);
			}
			this.patDemoWidget.tabPanel.setActiveTab(0);
			this.patDemoWidget.tabPanel.doLayout();

            //display all code gray alerts to user
            if (codeGrayAlerts.length > 0)
            {

                var displayCodeGray = function(idx, codeGrayAlertsArr)
                {
                     //display alert to user
                    Ext.Msg.alert(
                        codeGrayAlertsArr[idx].title,
                        codeGrayAlertsArr[idx].msg,
                        function(){
                            idx++;
                            if (idx < codeGrayAlertsArr.length)
                            {
                                displayCodeGray(idx, codeGrayAlertsArr);
                            }
                        }
                    );
                }

                displayCodeGray(0, codeGrayAlerts);
            }
	    }
	}
});
