/*
 * Janus 4.0 (c)
 * Copyright (c) 2011 Hawaii Resource Group LLC. All Rights Reserved.
 * Developed for the Pacific Telehealth & Technology Hui and the Pacific Joint Information Technology Center
 * Contributors:
 *             Honorable Senator Daniel K. Inouye
 *             VA Pacific Islands Health Care System
 *             Tripler Army Medical Center
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with the License.
 *
 * You may obtain a copy of the License at:
 *
 *            http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and limitations under the License.
 */

/**
 * Discharge Summaries Widget
 *
 * @param {} config
 *
 * @cfg patientToken
 *     Patient's security token
 * @cfg desktop
 *    Reference to application desktop object
 */
Ext.hui.widget.patient.DischargeSummaries = function(config)
{
    var today =new Date();
    var ONE_DAY = 24*60*60*1000;

//    if(!config.endDay){
        this.endDate = today;
//    } else {
//        this.endDate = new Date((today.getTime() + config.endDay*ONE_DAY));
//    }
//    if(!config.startDay){
        this.startDate = new Date((today.getTime() - 120*ONE_DAY));
//    } else {
//        this.startDate = new Date((today.getTime() + config.startDay*ONE_DAY));
//    }
    if(Ext.hui.app.EXTEND_YEAR=="true"){
                this.startDate.setYear(Ext.hui.app.EXTEND_YEAR_DATE);
            }

    Ext.hui.widget.patient.DischargeSummaries.superclass.constructor.call(this, config);
}

Ext.extend(Ext.hui.widget.patient.DischargeSummaries, Ext.hui.widget.Widget,
{
    title: "Discharge/Essentris Notes",
    height:200,
    layout:'border',
    tools:[
        {
			id: 'filter',
			qtip: 'Configure Filter',
            cls: 'x-tool-filter',
            overCls:'x-tool-filter-over',
			handler: function (event,toolEl, panel){
				if (panel.dateFilter) {
                    panel.dateFilter.toggleView();
				}
			}
		},
        {
            id:'help',
            qtip: 'Interface Status',
            handler: function(event, toolEl, panel)
            {
                var win = panel.desktop.createWindow({
                    title:"Interface Status ~ Discharge/Essentris Notes",
                    width: 600,
                    height: 500,
                    autoScroll: true
                });
                win.show();
                var sourceStatus = Ext.hui.app.BHIE_STATUS_NO_REPORT_MSG;

                if (panel.bhieSourceReport != null)
                {
                    sourceStatus = panel.bhieSourceReport;
                }

                win.body.dom.innerHTML = "<div class='bhie-report'>" + sourceStatus + "</div>";
            }
        },
        {
			id: 'copy',
			qtip: 'Copy to Clipboard',
            cls: 'x-tool-copy',
            overCls:'x-tool-copy-over',
			handler: function (event,toolEl, panel){
				if (panel.gridPanel) {
                    Ext.hui.common.copyGridPanel(panel.gridPanel);
				}
			}
		},
		{
			id: 'print',
			qtip: 'Print',
			handler: function (event,toolEl, panel){
				Ext.ux.Printer.BaseRenderer.prototype.stylesheetPath = 'css/print.css';
				if (panel.gridPanel) {
					Ext.ux.Printer.printPatInfo(panel.gridPanel);
				}
			}
		},
        {
            id:'refresh',
            qtip: 'Refresh',
            handler: function(event, toolEl, panel){
                panel.dataStore.reload();
            }
        },
        {
            id:'close',
            qtip: 'Close Widget',
            handler: function(event, toolEl, panel){
                panel.destroy();
                Ext.hui.session.saveConfig();
            }
        }
    ],
    initComponent: function() {

        this.bhieSourceReport = null;

        this.dataStore = new Ext.data.Store({
            restful:true,
            proxy: new Ext.data.HttpProxy({
                url: 'rest/patientDischargeSummaries.json',
                disableCaching:true,
                timeout: Ext.hui.common.CONNECTION_TIMEOUT_MS,
                method:'GET'
            }),
            baseParams:{
                patToken:this.patientToken,
                startDate:Ext.util.Format.dateRenderer("m/d/y")(this.startDate),
                endDate:Ext.util.Format.dateRenderer("m/d/y")(this.endDate)
            },
            reader: this.createJsonReader(),

            sortInfo: {
                field: 'noteDate',
                direction: 'DESC'
            },
            listeners:
            {
                exception:{
                    scope:this,
                    fn:function(dataProxy, type, action, options, response, arg){
                        Ext.hui.common.handleNetworkFailure(response.status, this);

                    }
                },
                beforeLoad:
                {
                    scope:this,
                    fn:function()
                    {
                        if (!this || this.isDestroy) return;
                        this.loading = new Ext.LoadMask(this.gridArea.el, {msg:"Loading...", store:this.dataStore});
                        this.loading.show();
                        if (this.dateFilter) this.dateFilter.disable();
                        this.bhieSourceReport = null;
                    }
                },
                load:
                {
                    scope:this,
                    fn:
                    function(store, records, options){
                        if (!this || this.isDestroy) return;

                        var jsonDt = [];
                        var bhieReportRecIdx = -1;
                        for(var i = 0; i < records.length; i++)
                        {
                            jsonDt[i] = records[i].json;
                            if (jsonDt[i].siteMoniker == Ext.hui.app.BHIE_STATUS_REPORT_FLAG)
                            {
                                this.bhieSourceReport = jsonDt[i].noteType;
                                bhieReportRecIdx = i;
                            }
                        }

                         //remove bhie report from list
                        if (bhieReportRecIdx > -1)
                        {
                           jsonDt.remove(jsonDt[bhieReportRecIdx]);
                           store.remove(records[bhieReportRecIdx]);
                        }

                        var inMemoryStore = new Ext.data.Store({
                            proxy: new Ext.ux.data.PagingMemoryProxy(jsonDt),
                            remoteSort:true,
                            reader: this.createJsonReader(),
                            sortInfo: {
                                field: 'noteDate',
                                direction: 'DESC'
                            }
                        });
                        //destroy previous grid panel
                        if (this.gridPanel)
                        {
                            this.gridPanel.destroy();
                        }
                        this.gridPanel = this.createGrid(inMemoryStore, this);
                        inMemoryStore.load({params:{start:0, limit:Ext.hui.common.WIDGET_NUM_RECORDS_PER_PAGE}});

                        this.gridArea.removeAll();
                        this.gridArea.add(this.gridPanel);
                        this.doLayout();

                        this.setTitle("Discharge/Essentris Notes ("+jsonDt.length+")");

                        if (this.dateFilter) this.dateFilter.enable();
                    }
                }

            }
        });

        this.dateFilter = new Ext.hui.filter.DateFilter({
            region:'north',
            dataStore:this.dataStore,
            startDate:this.startDate,
            endDate:this.endDate,
            disabled:true,
            hidden:true
        });


        this.gridArea = new Ext.Container({layout:'fit', region:'center', html:''});
        this.items = [this.dateFilter, this.gridArea];

        Ext.hui.widget.patient.DischargeSummaries.superclass.initComponent.call(this);
    },
    createJsonReader: function()
    {
        return new Ext.data.JsonReader({
                root: ''
            },
            [
                {name: 'noteType'},
                {name: 'noteDate', type:'date'},
                {name: 'recordDetailsToken'},
                {name: 'nonPlainTextNote'},
                {name: 'siteCode'},
                {name: 'siteMoniker'},
                {name: 'siteAgency'},
                {name: 'sourcePlatform'}
            ]);
    },
    createGrid: function(gridDataStore, refWidget)
    {
        var pagingBarItems = [
        {
            text: 'more >>',
            cls: 'more-link',
            listeners:
            {
                click: function(btn, pressed){
                    refWidget.desktop.createWindow({
                        patientToken:refWidget.patientToken,
                        desktop:refWidget.desktop,
                        startDate:refWidget.startDate,
                        endDate:refWidget.endDate
                    }, Ext.hui.desktopwindow.patient.DischargeSummaries).show();
                }

            }
        }];

        var dateRenderer = Ext.util.Format.dateRenderer('M d, Y H:i');

        return new Ext.grid.GridPanel({
            store: gridDataStore,
            columns: [
            {
                header   : 'Site',
                width    : 120,
                sortable : true,
                dataIndex: 'siteMoniker',
                renderer :
                function(value, metaData, record, rowIndex, colIndex, store) {
                    return Ext.hui.common.formatSiteMonikerDisplay(record.data.siteAgency, record.data.siteMoniker);
                }
            },
            {
                width    : 120,
                header   : 'Date',
                sortable : true,
                dataIndex: 'noteDate',
                renderer: {
                    fn:function(value, metaData, record, rowIndex, colIndex, store) {
                        var dateStr = dateRenderer(record.data.noteDate);

                        return dateStr;

                    },
                    scope:this
                }
            },
            {
                id: 'noteType',
                width    : 130,
                header   : 'Type',
                sortable : true,
                dataIndex: 'noteType',
                renderer :{
                        fn:function(value, metaData, record, rowIndex, colIndex, store) {
                            if (record.data.recordDetailsToken != null)
                            {
                                var noteType = record.data.noteType;
                                if (noteType.length < 1)
                                {
                                    noteType = "Note type unavailable, click link to view note";
                                }

                                 return String.format('<span class="record-details" id="dischargesummary-details-{0}-{1}">{2}</span>', refWidget.id, record.id, noteType);
                            }
                            else return record.data.noteType;

                        },
                        scope:this
                    }
            }
            ],
            listeners:{
                rowclick:
                {
                    scope:this,
                    fn:function(grid, rowIdx, evtObj) {
                        var record = grid.getStore().getAt(rowIdx);
                        if (Ext.get(String.format("dischargesummary-details-{0}-{1}", refWidget.id, record.id))){
                            if(evtObj.within(String.format("dischargesummary-details-{0}-{1}", refWidget.id, record.id)))
                            {
                                if (record.data.nonPlainTextNote)
                                {
                                    var noteURL = String.format("rest/patientDischargeSummaryDetails?patToken={0}&recToken={1}",
                                            refWidget.patientToken, record.data.recordDetailsToken);

                                    Ext.hui.common.openAuxBrowserWindow(noteURL, "IEHR_DISCHARGE_SUMMARY");
                                }
                                else
                                {
                                    refWidget.desktop.createWindow({
                                        title:String.format("Discharge/Essentris Note Details ~ {0} {1}",
                                                Ext.hui.common.formatSiteMonikerDisplay(record.data.siteAgency, record.data.siteMoniker),
                                                record.data.noteType),
                                        patientToken:refWidget.patientToken,
                                        desktop:refWidget.desktop,
                                        recordDetailsToken:record.data.recordDetailsToken
                                    }, Ext.hui.desktopwindow.patient.DischargeSummaryDetails).show();
                                }
                            }
                        }
                    }
                }
            },
            selModel: new Ext.grid.CellSelectionModel(),
            stripeRows: true,
            autoExpandColumn:'noteDate',
            border:true,
            bodyBorder:true,
            columnLines:true,
            loadMask:true,
            layout:'fit',
            region: 'center',
            bbar: new Ext.PagingToolbar({
                pageSize: Ext.hui.common.WIDGET_NUM_RECORDS_PER_PAGE,
                store: gridDataStore,
                displayInfo: true,
                items:pagingBarItems
            }),
            viewConfig: {
                emptyText:"No results",
                forceFit:true
            }
        });
    }
});

//Ext.reg('dischargesummaries_widget', Ext.hui.widget.patient.DischargeSummaries);

//SB: Using old alias to support widget creation from save data
Ext.reg('inpatientnotes_widget', Ext.hui.widget.patient.DischargeSummaries);