/*
 * Janus 4.0 (c)
 * Copyright (c) 2011 Hawaii Resource Group LLC. All Rights Reserved.
 * Developed for the Pacific Telehealth & Technology Hui and the Pacific Joint Information Technology Center
 * Contributors:
 *             Honorable Senator Daniel K. Inouye
 *             VA Pacific Islands Health Care System
 *             Tripler Army Medical Center
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with the License.
 *
 * You may obtain a copy of the License at:
 *
 *            http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and limitations under the License.
 */

/**
 * Provider Unsigned Notes Window
 * @param {} config Class configuration object
 * 
 * @cfg providerToken
 *     Provider's security token
 */
Ext.hui.desktopwindow.provider.UnsignedNotes = Ext.extend(Ext.hui.desktopwindow.DesktopWindow,
{
	initComponent: function() {

        this.sortInfo =  {
            field: 'noteDate',
            direction: 'DESC'
        };

        this.dataStore = new Ext.data.Store({
			restful:true,
			proxy: new Ext.data.HttpProxy({
						url: 'rest/providerUnsignedNotes.json',
						disableCaching:true,
						method:'GET',
                        timeout: Ext.hui.common.CONNECTION_TIMEOUT_MS
	    			}),
            baseParams:
            {
                provToken:this.provToken,
                startDate:Ext.util.Format.dateRenderer("m/d/y")(this.startDate),
                endDate:Ext.util.Format.dateRenderer("m/d/y")(this.endDate)
            },
	        reader: this.createJsonReader(),
	        
	        sortInfo: this.sortInfo,
			listeners:
			{
				exception:{scope:this, fn:function(dataProxy, type, action, options, response, arg){
												Ext.hui.common.handleNetworkFailure(response.status, this);
                }},
                beforeLoad:
                {
                    scope:this,
                    fn:function()
                    {
                        if (!this || this.isDestroy) return;
                        this.loading = new Ext.LoadMask(this.gridArea.el, {msg:"Loading...", store:this.dataStore});
                        this.loading.show();
                        if (this.filter) this.filter.disable();
                    }
                },
				load:
				{
					scope:this, 
					fn:
					  	function(store, records, options){
                            if (!this || this.isDestroy) return;
                            if (this.filter)
                            {
                                this.filter.enable();
                                this.filter.loadWithData();
                            }
                            this.setTitle("Unsigned Notes ("+records.length+")");
	    				}
    			},
                datachanged:
                {
                    scope:this,
                    fn:function(store)
                    {
                        if (!this || this.isDestroy) return;
                        this.tableExpandToggleText = Ext.hui.common.EXPAND_TABLE_TEXT;

                        var recs = store.getRange();
                        var jsonDt = [];
                        for(var i = 0; i < recs.length; i++)
                        {
                            jsonDt[i] = recs[i].json;
                        }

                        var inMemoryStore = new Ext.data.Store({
                            proxy: new Ext.ux.data.PagingMemoryProxy(jsonDt),
                            remoteSort:true,
                            reader: this.createJsonReader(),
                            sortInfo: this.sortInfo
                        });

                        //destroy previous grid panel
                        if (this.gridPanel)
                        {
                            this.gridPanel.destroy();
                        }
                        this.gridPanel = this.createGrid(inMemoryStore, this, Ext.hui.common.WINDOW_NUM_RECORDS_PER_PAGE);
                        inMemoryStore.load({params:{start:0, limit:Ext.hui.common.WINDOW_NUM_RECORDS_PER_PAGE}});

                        this.gridArea.removeAll();
                        this.gridArea.add(this.gridPanel);
                        this.doLayout();
                        if (this.filter) this.filter.displayFilter(true);
                    }
                }
			},
	        autoLoad: true
	    });

        this.filter = new Ext.hui.filter.NoteFilter({
            region:'north',
            dataStore:this.dataStore,
            startDate:this.startDate,
            endDate:this.endDate,
            disabled:true,
            hidden:true
        });

	    this.title='Unsigned Notes';
        this.width = 900;
        this.height = 500;
        this.layout = "border";
        this.iconCls = 'icon-grid';
        this.animCollapse = false;
        this.constrainHeader = true;
        this.draggable = true;
        this.dataStore = this.dataStore;
	    this.tools = 
	    [{
			id: 'filter',
			qtip: 'Configure Filter',
            cls: 'x-tool-filter',
            overCls:'x-tool-filter-over',
			handler: function (event,toolEl, panel){
				if (panel.filter) {
                    panel.filter.toggleView();
				}
			}
		},
        {
			id: 'copy',
			qtip: 'Copy to Clipboard',
			handler: function (event,toolEl, panel){
				if (panel.gridPanel) {
					 Ext.hui.common.copyGridPanel(panel.gridPanel);
				}
			}
		},
		{
			id: 'print',
			qtip: 'Print',
			handler: function (event,toolEl, panel){
				Ext.ux.Printer.BaseRenderer.prototype.stylesheetPath = 'css/print.css';
				if (panel.gridPanel) {
					Ext.ux.Printer.printPatInfo(panel.gridPanel);
				}
			}
		},
			{
			    id:'refresh',
			    qtip: 'Refresh',
			    handler: function(event, toolEl, win){
			        win.dataStore.reload();
			    }
			}
		];

        this.gridArea = new Ext.Container({layout:'fit', region:'center', html:''});

	    this.items = [this.filter, this.gridArea];
	    
	    Ext.hui.desktopwindow.provider.UnsignedNotes.superclass.initComponent.call(this);
	},

    createJsonReader: function()
    {
        return new Ext.data.JsonReader(
        {
            root: ''
        },
        [
            {name: 'noteType'},
            {name: 'noteDate', type:'date'},
            {name: 'recordDetailsToken'},
            {name: 'nonPlainTextNote'},
            {name: 'siteCode'},
            {name: 'siteMoniker'},
            {name: 'patientName'},
            {name: 'patientToken'},
            {name: 'siteAgency'},
            {name: 'sourcePlatform'}
        ])
    },

    createGrid: function(gridDataStore, refWin, recordsPerPage)
    {
        var dateRenderer = Ext.util.Format.dateRenderer('M d, Y H:i');

		var gridPanel = new Ext.grid.GridPanel({
	        store: gridDataStore,
	        columns: [
	            {
                header   : 'Site',
                width    : 80,
                sortable : true,
                dataIndex: 'siteMoniker',
                renderer :
                function(value, metaData, record, rowIndex, colIndex, store) {
                    return Ext.hui.common.formatSiteMonikerDisplay(record.data.siteAgency, record.data.siteMoniker);
                }
            },
            {
                width    : 150,
                header   : 'Date',
                sortable : true,
                dataIndex: 'noteDate',
                renderer: dateRenderer
            },
            {
                width    : 200,
                header   : 'Patient',
                sortable : true,
                dataIndex: 'patientName',
                renderer :
                        function(value, metaData, record, rowIndex, colIndex, store) {
                            return String.format('<span id="patient-link-{0}-{1}" class="selectpatient">'+record.data.patientName+'</span>', refWin.id, record.id);
                        }
            },
            {
                id: 'noteType',
                width    : 150,
                header   : 'Type',
                sortable : true,
                dataIndex: 'noteType',
                renderer :{
                    fn:function(value, metaData, record, rowIndex, colIndex, store) {
                        if (record.data.recordDetailsToken != null)
                        {
                            return String.format('<span class="record-details" id="unsignednote-details-{0}-{1}">{2}</span>', this.id, record.id, record.data.noteType);
                        }
                        else return record.data.noteType;

                    },
                    scope:this
                }
            }
	        ],
	        listeners:{
	        	rowclick:
	        	{
	        		scope:this,
	        		fn:function(grid, rowIdx, evtObj) {
	        			var record = grid.getStore().getAt(rowIdx);
                        if(evtObj.within(String.format("patient-link-{0}-{1}", refWin.id, record.id)))
                        {
                            Ext.hui.app.selectPatient(record.data.patientToken, record.data.patientName, refWin.id);
                        }
                        if (Ext.get(String.format("unsignednote-details-{0}-{1}", refWin.id, record.id))){
                            if(evtObj.within(String.format("unsignednote-details-{0}-{1}", refWin.id, record.id)))
                            {
                                if (record.data.nonPlainTextNote)
                                {
                                    var noteURL = String.format("rest/rest/patientProgressNoteDetails?patToken={0}&recToken={1}",
                                            record.data.patientToken, record.data.recordDetailsToken);

                                    Ext.hui.common.openAuxBrowserWindow(noteURL, "IEHR_UNSIGNED_NOTE");
                                }
                                else
                                {
                                    refWin.desktop.createWindow({
                                        title:String.format("Unsigned Note Details ~ {0} {1} :: {2}",
                                                Ext.hui.common.formatSiteMonikerDisplay(record.data.siteAgency, record.data.siteMoniker),
                                                record.data.noteType, record.data.patientName),
                                        patientToken:record.data.patientToken,
                                        desktop:refWin.desktop,
                                        recordDetailsToken:record.data.recordDetailsToken
                                    }, Ext.hui.desktopwindow.patient.ProgressNoteDetails).show();
                                }
                            }
                        }
	        		}
	        	},
               viewready:
                {
                    scope:refWin,
                    fn:function()
                    {
                        if (!this || this.isDestroy) return;
                        if (this.filter) this.filter.enable();
                        this.loading.hide();
                    }
                }
	        },
	        stripeRows: true,
            autoExpandColumn:'noteType',
            border:true,
            bodyBorder:true,
            loadMask:true,
            layout:'fit',
            region: 'center',
            columnLines:true,
            bbar: refWin.createPagingToolbar(gridDataStore, recordsPerPage),
            viewConfig: {
                emptyText:"No results",
                forceFit:true
            }
	    });

        return gridPanel;
    }
});

Ext.reg('prov_unsignednotes_window', Ext.hui.desktopwindow.provider.UnsignedNotes);