/*
 * Janus 4.0 (c)
 * Copyright (c) 2011 Hawaii Resource Group LLC. All Rights Reserved.
 * Developed for the Pacific Telehealth & Technology Hui and the Pacific Joint Information Technology Center
 * Contributors:
 *             Honorable Senator Daniel K. Inouye
 *             VA Pacific Islands Health Care System
 *             Tripler Army Medical Center
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with the License.
 *
 * You may obtain a copy of the License at:
 *
 *            http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and limitations under the License.
 */

/**
 * Vitals Table Window
 * @param {} config Class configuration object
 *
 */
Ext.hui.desktopwindow.patient.VitalsTable = Ext.extend(Ext.hui.desktopwindow.DesktopWindow,
{
    initComponent: function() {
		if (!this.title || this.title.length < 0)
        {
            this.title = "Vitals Table";
        }
	    this.width = 800;
        this.height = 510;
        this.cls = "vitalstable";
        this.autoScroll = true;

        this.on('afterrender', function(){this.fetchData()}, this);

        this.addTool({
            id: 'copy',
			qtip: 'Copy to Clipboard',
            cls: 'x-tool-copy',
            overCls:'x-tool-copy-over',
			handler: function (event,toolEl, window){
				if (window) {
                    Ext.hui.common.copyWindow(window);
				}
			}
        });
        this.addTool({
            id: 'print',
			qtip: 'Print',
			handler: function (event,toolEl, window){
				Ext.ux.Printer.BaseRenderer.prototype.stylesheetPath = 'css/print.css';
				if (window) {
					Ext.ux.Printer.printPatInfo(window);
				}
			}
        });
	    Ext.hui.desktopwindow.patient.VitalsTable.superclass.initComponent.call(this);
	},

	fetchData: function()
	{
		this.getUpdater().setRenderer(new Ext.hui.desktopwindow.patient.VitalsTableRenderer(this));
		this.load({
			url: 'rest/patientVitalsHistogram.json',
			method:'GET',
			params: {
                patToken:this.patientToken,
                recToken:this.histogramRecordToken
            },
			discardUrl: true,
		    nocache: true,
            disableCaching:true,
		    text: 'Loading...',
		    timeout:Ext.hui.common.CONNECTION_TIMEOUT_SECONDS,
		    scripts: false,
		    scope:this,
	    	callback:function(options, isSuccess, response)
			{
				if (!isSuccess)
				{
					Ext.hui.common.handleNetworkFailure(response.status, this);
				}
			}
		});
	}

});

Ext.reg('vitalstable_window', Ext.hui.desktopwindow.patient.VitalsTable);

/**
 * Vitals table renderer
 *
 * @param window The window to render into.
 *
 */
Ext.hui.desktopwindow.patient.VitalsTableRenderer = function(window)
{
	this.window = window;
	Ext.hui.desktopwindow.patient.VitalsTableRenderer.superclass.constructor.call(this);
}

Ext.extend(Ext.hui.desktopwindow.patient.VitalsTableRenderer, Ext.Updater.BasicRenderer,
{
    render: function(el, xhr, updateManager, callback)
	{
        if (!this.window || this.window.isDestroy) return;
        var jsonResponse = Ext.util.JSON.decode(xhr.responseText);

		if (jsonResponse.vitalEntries == null || jsonResponse.vitalEntries.length < 1)
        {
            this.window.body.dom.innerHTML = '<div class="errortext">An error occurred when attempting to retrieve table data for this vital result series.</div>';
        }
        else
        {
            if (jsonResponse.bloodPressureVital)
            {
                this.renderBloodPressureTable(jsonResponse, window);
            }
            else
            {
                this.renderNonBloodPressureTable(jsonResponse, window);
            }
        }

        this.window.doLayout();
	},

    renderBloodPressureTable: function(jsonResponse, window)
    {
        var vitalsTableDivId = "vitalstable_" + this.window.id;
        var html = "";
        var generateData = function(records)
        {
            var data = [];

            for(var i = 0; i < records.length; i++)
            {
                var vitalResult = records[i];
                data.push
                ({
                    'date':Ext.util.Format.dateRenderer('M d, Y H:i')(vitalResult.data.dateTimeTaken),
                    'systolic':parseFloat(vitalResult.data.systolicRate),
                    'diastolic':parseFloat(vitalResult.data.diastolicRate),
                    'units': vitalResult.units,
                    'qualifiers': vitalResult.qualifiers,
                    'siteCode':vitalResult.data.siteCode,
                    'siteMoniker':vitalResult.data.siteMoniker,
                    'siteAgency':vitalResult.data.siteAgency,
                    'sourcePlatform':vitalResult.data.sourcePlatform
                });
            }

            return data;
        }

        var dataStore = new Ext.data.JsonStore({
                fields: [ {name:'dateTimeTaken', type:'date'}, 'systolicRate', 'diastolicRate', 'units', 'qualifiers', 'siteCode', 'siteMoniker', 'siteAgency', 'sourcePlatform'],
                data:jsonResponse.vitalEntries,
                sortInfo: {
                    field: 'dateTimeTaken',
                    direction: 'ASC'
                }
            });

        var tpl = new Ext.XTemplate
        (
             '<div class="vitalstable">',
                '<div><b>Type:</b>&nbsp;{vitalsType}<br/><b>Units:</b>&nbsp;{vitalsUnits}',
                '<div id="disclaimer_{vitalsTableDivId}" class="disclaimer"></div>',
                '<table width="100%">',
                    '<thead>',
                        '<tr>',
                            '<th scope="col" class="table-header site">Site</th>',
                            '<th scope="col" class="table-header date">Date Taken</th>',
                            '<th scope="col" class="table-header rate">Systolic</th>',
                            '<th scope="col" class="table-header rate">Diastolic</th>',
                         '</tr>',
                    '</thead>',
                    '<tbody>',
                        '<tpl for="vitalsTableData">',
                        '<tr class="{[xindex % 2 === 0 ? "even" : "odd"]}">',
                            '<td width="10%">{[values.siteAgency == Ext.hui.app.AGENCY_DOD ? "<img src=\'images/dodMarker.png\'  alt=\'DoD Marker\' />" : "<img src=\'images/vaMarker.png\'  alt=\'VA Marker\'/>"]}&nbsp;{siteMoniker}</td>',
                            '<td width="20%">{date}</td>',
                            '<td width="25%">{systolic}</td>',
                            '<td width="25%">{diastolic}</td>',
                        '</tr>',
                        '</tpl>',
                    '</tbody>',
                '</table>',
                '<br/>',
                '<div><b>Minimum Systolic:</b> {minSystolic} - {minSystolicDateFormatted}</div>',
                '<div><b>Maximum Systolic:</b> {maxSystolic} - {maxSystolicDateFormatted}</div>',
                '<br/>',
                '<div><b>Minimum Diastolic:</b> {minSystolic} - {minDiastolicDateFormatted}</div>',
                '<div><b>Maximum Diastolic:</b> {maxSystolic} - {maxDiastolicDateFormatted}</div>',
                '<br/>',
                '<div><b>Number of Results:</b> {count}</div>',
                '<br/>',
                '<div><b>Average Systolic:</b> {avgSystolic}</div>',
                '<div><b>Average Diastolic:</b> {avgDiastolic}</div>',
                '<br/>',
            '</div>'
        );

        var firstResult = jsonResponse.vitalEntries[0];
        jsonResponse.vitalsTableData = generateData(dataStore.getRange());
        jsonResponse.vitalsType = firstResult.type;
        jsonResponse.vitalsUnits = firstResult.units;
        jsonResponse.vitalsQualifiers = firstResult.qualifiers;
        jsonResponse.vitalsTableDivId = vitalsTableDivId;
        jsonResponse.minSystolicDateFormatted = Ext.util.Format.dateRenderer('M d, Y H:i')(new Date(jsonResponse.minSystolicDate))
        jsonResponse.maxSystolicDateFormatted = Ext.util.Format.dateRenderer('M d, Y H:i')(new Date(jsonResponse.maxSystolicDate))
        jsonResponse.minDiastolicDateFormatted = Ext.util.Format.dateRenderer('M d, Y H:i')(new Date(jsonResponse.minDiastolicDate))
        jsonResponse.maxDiastolicDateFormatted = Ext.util.Format.dateRenderer('M d, Y H:i')(new Date(jsonResponse.maxDiastolicDate))
        html = tpl.apply(jsonResponse);

        this.window.body.dom.innerHTML = html;

        if (jsonResponse.exceptionFlag == true)
        {
            Ext.fly('disclaimer_'+vitalsTableDivId).dom.innerHTML = "<b>Note: Only numeric results have been displayed. Other results are available that have not been numerically resulted.</b>";
        }
    },

    renderNonBloodPressureTable: function(jsonResponse, window)
    {
        var vitalsTableDivId = "vitalstable_" + this.window.id;
        var html = "";
        var generateData = function(records)
        {
            var data = [];

            for(var i = 0; i < records.length; i++)
            {
                var vitalResult = records[i];
                data.push
                ({
                    'date':Ext.util.Format.dateRenderer('M d, Y H:i')(vitalResult.data.dateTimeTaken),
                    'rate':parseFloat(vitalResult.data.rate),
                    'units':vitalResult.data.units,
                    'qualifiers':vitalResult.data.qualifiers,
                    'siteCode':vitalResult.data.siteCode,
                    'siteMoniker':vitalResult.data.siteMoniker,
                    'siteAgency':vitalResult.data.siteAgency,
                    'sourcePlatform':vitalResult.data.sourcePlatform
                });
            }

            return data;
        }

        var dataStore = new Ext.data.JsonStore({
                fields: [ {name:'dateTimeTaken', type:'date'}, 'rate', 'units', 'qualifiers', 'siteCode', 'siteMoniker', 'siteAgency', 'sourcePlatform'],
                data:jsonResponse.vitalEntries,
                sortInfo: {
                    field: 'dateTimeTaken',
                    direction: 'ASC'
                }
            });

        var tpl = new Ext.XTemplate
        (
             '<div class="vitalstable">',
                '<div><b>Type:</b>&nbsp;{vitalsType}<br/><b>Units:</b>&nbsp;{vitalsUnits}',
                '<div id="disclaimer_{vitalsTableDivId}" class="disclaimer"></div>',
                '<table width="100%">',
                    '<thead>',
                        '<tr>',
                            '<th scope="col" class="table-header site">Site</th>',
                            '<th scope="col" class="table-header date">Date Taken</th>',
                            '<th scope="col" class="table-header rate">Result</th>',
                         '</tr>',
                    '</thead>',
                    '<tbody>',
                        '<tpl for="vitalsTableData">',
                        '<tr class="{[xindex % 2 === 0 ? "even" : "odd"]}">',
                            '<td width="10%">{[values.siteAgency == Ext.hui.app.AGENCY_DOD ? "<img src=\'images/dodMarker.png\' alt=\'DoD Marker\'/>" : "<img src=\'images/vaMarker.png\'  alt=\'VA Marker\'/>"]}&nbsp;{siteMoniker}</td>',
                            '<td width="25%">{date}</td>',
                            '<td width="25%">{rate}</td>',
                        '</tr>',
                        '</tpl>',
                    '</tbody>',
                '</table>',
                '<br/>',
                '<div><b>Minimum:</b> {minRate} - {minRateDateFormatted}</div>',
                '<div><b>Maximum:</b> {maxRate} - {maxRateDateFormatted}</div>',
                '<div><b>Number of Results:</b> {count}</div>',
                '<div><b>Average:</b> {avgRate}</div>',
                '<br/>',
            '</div>'
        );

        var firstResult = jsonResponse.vitalEntries[0];
        jsonResponse.vitalsTableData = generateData(dataStore.getRange());
        jsonResponse.vitalsType = firstResult.type;
        jsonResponse.vitalsUnits = firstResult.units;
        jsonResponse.vitalsTableDivId = vitalsTableDivId;
        jsonResponse.minRateDateFormatted = Ext.util.Format.dateRenderer('M d, Y H:i')(new Date(jsonResponse.minRateDate))
        jsonResponse.maxRateDateFormatted = Ext.util.Format.dateRenderer('M d, Y H:i')(new Date(jsonResponse.maxRateDate))
        html = tpl.apply(jsonResponse);

        this.window.body.dom.innerHTML = html;

        if (jsonResponse.exceptionFlag == true)
        {
            Ext.fly('disclaimer_'+vitalsTableDivId).dom.innerHTML = "<b>Note: Only numeric results have been displayed. Other results are available that have not been numerically resulted.</b>";
        }
    }
});
