/*
 * Janus 4.0 (c)
 * Copyright (c) 2011 Hawaii Resource Group LLC. All Rights Reserved.
 * Developed for the Pacific Telehealth & Technology Hui and the Pacific Joint Information Technology Center
 * Contributors:
 *             Honorable Senator Daniel K. Inouye
 *             VA Pacific Islands Health Care System
 *             Tripler Army Medical Center
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with the License.
 *
 * You may obtain a copy of the License at:
 *
 *            http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and limitations under the License.
 */

/**
 * Vitals Graph Window
 * @param {} config Class configuration object
 *
 * @cfg patientToken
 *     Patient's security token
 * @cfg histogramRecordToken
 * 		Histogram security token
 */
Ext.hui.desktopwindow.patient.VitalsGraph = Ext.extend(Ext.hui.desktopwindow.DesktopWindow,
{
    initComponent: function() {
		if (!this.title || this.title.length < 0)
        {
            this.title = "Vitals Graph";
        }
	    this.width = 900;
        this.height = 600;
        this.cls = "vitalsgraph";
        this.autoScroll = true;

        this.on('afterrender', function(){this.fetchData()}, this);

        this.addTool({
            id: 'print',
			qtip: 'Print',
			handler: function (event,toolEl, window){
				Ext.ux.Printer.BaseRenderer.prototype.stylesheetPath = 'css/print.css';
				if (window) {
					Ext.ux.Printer.printPatInfo(window);
				}
			}
        });
	    Ext.hui.desktopwindow.patient.VitalsGraph.superclass.initComponent.call(this);
	},

	fetchData: function()
	{
		this.getUpdater().setRenderer(new Ext.hui.desktopwindow.patient.VitalsGraphRenderer(this));
		this.load({
			url: 'rest/patientVitalsHistogram.json',
			method:'GET',
			params: {
                patToken:this.patientToken,
                recToken:this.histogramRecordToken
            },
			discardUrl: true,
		    nocache: true,
            disableCaching:true,
		    text: 'Loading...',
		    timeout:Ext.hui.common.CONNECTION_TIMEOUT_SECONDS,
		    scripts: false,
		    scope:this,
	    	callback:function(options, isSuccess, response)
			{
				if (!isSuccess)
				{
					Ext.hui.common.handleNetworkFailure(response.status, this);
				}
			}
		});
	}

});

Ext.reg('vitalsgraph_window', Ext.hui.desktopwindow.patient.VitalsGraph);

/**
 * Problem details renderer, parses json response and sends it to the widgets initLayout function.
 */
Ext.hui.desktopwindow.patient.VitalsGraphRenderer = function(window)
{
	this.window = window;
	Ext.hui.desktopwindow.patient.VitalsGraphRenderer.superclass.constructor.call(this);
}

Ext.extend(Ext.hui.desktopwindow.patient.VitalsGraphRenderer, Ext.Updater.BasicRenderer,
{
    render: function(el, xhr, updateManager, callback)
	{
        if (!this.window || this.window.isDestroy) return;

        var jsonResponse = Ext.util.JSON.decode(xhr.responseText);

        if (jsonResponse.vitalEntries == null || jsonResponse.vitalEntries.length < 1)
        {
            this.window.body.dom.innerHTML = '<div class="errortext">An error occurred when attempting to retrieve data for this vital result series.</div>';
        }
        else
        {
            if (jsonResponse.bloodPressureVital)
            {
                this.renderBloodPressureGraph(jsonResponse, window);
            }
            else
            {
                this.renderNonBloodPressureGraph(jsonResponse, window);
            }
        }
        this.window.doLayout();
	},

    renderBloodPressureGraph: function(jsonResponse, window)
    {
        var vitalsGraphDivId = "vitalsgraph_" + this.window.id;
        var html = "";
        var generateData = function(records)
        {
            var data = {systolic:[], diastolic:[]};

            for(var i = 0; i < records.length; i++)
            {
                var vitalResult = records[i];

                //flot graph lib always displays timestamps according to UTC

                data.systolic.push(
                    [
                       Ext.hui.common.createUTCDate(vitalResult.data.dateTimeTaken).getTime(),
                       parseFloat(vitalResult.data.systolicRate)
                    ]
                );

                data.diastolic.push(
                    [
                       Ext.hui.common.createUTCDate(vitalResult.data.dateTimeTaken).getTime(),
                       parseFloat(vitalResult.data.diastolicRate)
                    ]
                );
            }

            return data;
        }

        var tpl = new Ext.XTemplate
        (
             '<div class="vitalsgraphContainer">',
                '<div><b>Type:</b>&nbsp;{vitalsType}<br/><b>Units:</b>&nbsp;{vitalsUnits}',
                '<div class="graph-container" id="{vitalsGraphDivId}">',
                '</div>',
                '<div class="legendContainer">',
                    '<div class="legend">',
                        '<div class="title">Graph Legend</div>',
                        '<div class="entry"><span class="systolic-color"></span><span>&nbsp;systolic</span></div>',
                        '<div class="entry"><span class="diastolic-color"></span><span>&nbsp;diastolic</span></div>',
                    '</div>',
                '</div>',
                '<div id="zoomNav-instructions">Use mouse wheel to zoom, click and drag to pan.</div>',
                '<div id="disclaimer_{vitalsGraphDivId}" class="disclaimer"></div>',
                '<div id="graphZoomNavContainer" class="zoomNavContainer">',
                    '<div id="zoomNav-{vitalsGraphDivId}"></div>',
                '</div>',
                '<div>',
                    '<div class="graphdata">',
                        '<div><b>Minimum Systolic:</b> {minSystolic} - {minSystolicDateFormatted}</div>',
                        '<div><b>Maximum Systolic:</b> {maxSystolic} - {maxSystolicDateFormatted}</div>',
                        '<br/>',
                        '<div><b>Minimum Diastolic:</b> {minDiastolic} - {minDiastolicDateFormatted}</div>',
                        '<div><b>Maximum Diastolic:</b> {maxDiastolic} - {maxDiastolicDateFormatted}</div>',
                        '<br/>',
                        '<div><b>Number of Results:</b> {count}</div>',
                        '<br/>',
                        '<div><b>Average Systolic:</b> {avgSystolic}</div>',
                        '<div><b>Average Diastolic:</b> {avgDiastolic}</div>',
                     '</div>',
                '</div>',
            '</div>'
        );

        var firstResult = jsonResponse.vitalEntries[0];

        jsonResponse.vitalsType = firstResult.type;
        jsonResponse.vitalsUnits = firstResult.units;
        jsonResponse.vitalsGraphDivId = vitalsGraphDivId;
        jsonResponse.minSystolicDateFormatted = Ext.util.Format.dateRenderer('M d, Y H:i')(new Date(jsonResponse.minSystolicDate))
        jsonResponse.maxSystolicDateFormatted = Ext.util.Format.dateRenderer('M d, Y H:i')(new Date(jsonResponse.maxSystolicDate))
        jsonResponse.minDiastolicDateFormatted = Ext.util.Format.dateRenderer('M d, Y H:i')(new Date(jsonResponse.minDiastolicDate))
        jsonResponse.maxDiastolicDateFormatted = Ext.util.Format.dateRenderer('M d, Y H:i')(new Date(jsonResponse.maxDiastolicDate))
        html = tpl.apply(jsonResponse);

        this.window.body.dom.innerHTML = html;

        if (jsonResponse.exceptionFlag == true)
        {
            Ext.fly('disclaimer_'+vitalsGraphDivId).dom.innerHTML = "<b>Note: Only numeric results have been displayed. Other results are available that have not been numerically resulted.</b>";
        }

        var dataStore = new Ext.data.JsonStore({
                fields: [ {name:'dateTimeTaken', type:'date'}, 'systolicRate', 'diastolicRate', 'units', 'qualifiers', 'siteCode', 'siteMoniker', 'siteAgency', 'sourcePlatform'],
                data:jsonResponse.vitalEntries,
                sortInfo: {
                    field: 'dateTimeTaken',
                    direction: 'ASC'
                }
            });

        var dataVals = generateData(dataStore.getRange());

        var plot = $.plot($("#"+vitalsGraphDivId), [{data: dataVals.systolic, label:"systolic"}, {data: dataVals.diastolic, label:"diastolic"}],
        {
            xaxis:
            {
                mode: "time",
                timeformat: "%b %0d, %y"
            },
            series:
            {
                lines: { show: true },
                points: { show: true }
            },
            grid:
            {
                hoverable: true,
                clickable: true
            },
            zoom:
            {
                interactive: true
            },
            pan:
            {
                interactive: true
            },
            legend:{show:false},
            colors:["#426ab3", "#d09b2c"]
        });

        // add zoom out button
        $('<div role="button" class="button zoom-buttons", style="top:20px">zoom out</div>').appendTo($("#zoomNav-"+vitalsGraphDivId)).click(function (e) {
            e.preventDefault();
            plot.zoomOut();
        });

        // little helper for taking the repetitive work out of placing
        // panning arrows
        var addArrow = function(dir, right, top, offset) {
            $('<span role="button" class="button pan-arrow-'+dir+'" style="right:' + right + 'px;top:' + top + 'px"></span>').appendTo($("#zoomNav-"+vitalsGraphDivId)).click(function (e) {
                e.preventDefault();
                plot.pan(offset);
            });
        }

        addArrow('left', 55, 60, { left: -100 });
        addArrow('right', 25, 60, { left: 100 });
        addArrow('up', 40, 45, { top: -100 });
        addArrow('down', 40, 75, { top: 100 });

        // add zoom out button
        $('<div role="button" class="button zoom-buttons" style="top:100px;">zoom in</div>').appendTo($("#zoomNav-"+vitalsGraphDivId)).click(function (e) {
            e.preventDefault();
            plot.zoom();
        });

        var showGraphTooltip = function(x, y, contents) {
            $('<div id="'+vitalsGraphDivId+'_tooltip">' + contents + '</div>').css( {
                position: 'absolute',
                display: 'none',
                'text-align':'center',
                top: y-80,
                left: x-50,
                border: '1px solid #fdd',
                padding: '2px',
                'background-color': '#fee',
                opacity: 0.80
            }).appendTo("body").fadeIn(200);
        }
        var me = this;
        var previousPoint = null;
        $("#"+vitalsGraphDivId).bind("plothover", function (event, pos, item) {

            if (item) {
                if (previousPoint != item.dataIndex) {
                    previousPoint = item.dataIndex;

                    $("#"+vitalsGraphDivId+"_tooltip").remove();
                    var x = item.datapoint[0],
                        y = item.datapoint[1];

                    var records = dataStore.getRange();
                    var idx = item.dataIndex;
                    var record = records[idx];

                    //ensure index matches data

                    var label = item.series.label;
                    if ((label == "systolic" && record.data.systolicRate == y) ||
                            (label == "diastolic" && record.data.diastolicRate == y))
                    {
                        //flot graph lib always displays timestamps according to UTC
                        if (Ext.hui.common.createUTCDate(record.data.dateTimeTaken).getTime() == x)
                        {
                            showGraphTooltip(item.pageX, item.pageY,
                                Ext.hui.common.formatSiteMonikerDisplay(records[idx].data.siteAgency, records[idx].data.siteMoniker) +
                                "<br/>"+
                                record.data.systolicRate + "/" + record.data.diastolicRate + " " + record.data.units +
                                "<br/>" +
                                Ext.util.Format.dateRenderer('M d, Y H:i')(records[idx].data.dateTimeTaken));
                        }
                    }
                }
            }
            else {
                $("#"+vitalsGraphDivId+"_tooltip").remove();
                previousPoint = null;
            }
        });
    },

    renderNonBloodPressureGraph: function(jsonResponse, window)
    {
        var vitalsGraphDivId = "vitalsgraph_" + this.window.id;
        var html = "";
        var generateData = function(records)
        {
            var data = [];

            for(var i = 0; i < records.length; i++)
            {
                var vitalResult = records[i];

                data.push
                (
                   [
                       //flot graph lib always displays timestamps according to UTC
                       Ext.hui.common.createUTCDate(vitalResult.data.dateTimeTaken).getTime(),
                       parseFloat(vitalResult.data.rate)
                   ]

                );
            }

            return data;
        }

        var tpl = new Ext.XTemplate
        (
            '<div class="vitalsgraphContainer">',
                '<div><b>Type:</b>&nbsp;{vitalsType}<br/><b>Units:</b>&nbsp;{vitalsUnits}',
                '<div class="graph-container" id="{vitalsGraphDivId}"></div>',
                '<div id="zoomNav-instructions">Use mouse wheel to zoom, click and drag to pan.</div>',
                '<div id="disclaimer_{vitalsGraphDivId}" class="disclaimer"></div>',
                '<div id="graphZoomNavContainer" class="zoomNavContainer">',
                    '<div id="zoomNav-{vitalsGraphDivId}"></div>',
                '</div>',
                '<div><b>Minimum:</b> {minRate} - {minRateDateFormatted}</div>',
                '<div><b>Maximum:</b> {maxRate} - {maxRateDateFormatted}</div>',
                '<div><b>Number of Results:</b> {count}</div>',
                '<div><b>Average:</b> {avgRate}</div>',
             '</div>'
        );

        var firstResult = jsonResponse.vitalEntries[0];

        jsonResponse.vitalsType = firstResult.type;
        jsonResponse.vitalsUnits = firstResult.units;
        jsonResponse.vitalsGraphDivId = vitalsGraphDivId;
        jsonResponse.minRateDateFormatted = Ext.util.Format.dateRenderer('M d, Y H:i')(new Date(jsonResponse.minRateDate))
        jsonResponse.maxRateDateFormatted = Ext.util.Format.dateRenderer('M d, Y H:i')(new Date(jsonResponse.maxRateDate))
        html = tpl.apply(jsonResponse);

//        this.window.on("afterrender", function(){
//            alert("afterrender");
//        });
        this.window.body.dom.innerHTML = html;


        if (jsonResponse.exceptionFlag == true)
        {
            Ext.fly('disclaimer_'+vitalsGraphDivId).dom.innerHTML = "<b>Note: Only numeric results have been displayed. Other results are available that have not been numerically resulted.</b>";
        }

        var dataStore = new Ext.data.JsonStore({
            fields: [ {name:'dateTimeTaken', type:'date'}, 'rate', 'units', 'qualifiers', 'siteCode', 'siteMoniker', 'siteAgency', 'sourcePlatform'],
            data:jsonResponse.vitalEntries,
            sortInfo: {
                field: 'dateTimeTaken',
                direction: 'ASC'
            }
        });

        var dataVals = generateData(dataStore.getRange());

        var plot = $.plot($("#"+vitalsGraphDivId), [ dataVals],
        {
            xaxis:
            {
                mode: "time",
                timeformat: "%b %0d, %y"
            },
            series:
            {
                lines: { show: true },
                points: { show: true }
            },
            grid:
            {
                hoverable: true,
                clickable: true
            },
            zoom:
            {
                interactive: true
            },
            pan:
            {
                interactive: true
            }

        });

        // add zoom out button
        $('<div role="button" class="button zoom-buttons", style="top:20px">zoom out</div>').appendTo($("#zoomNav-"+vitalsGraphDivId)).click(function (e) {
            e.preventDefault();
            plot.zoomOut();
        });

        // little helper for taking the repetitive work out of placing
        // panning arrows
        var addArrow = function(dir, right, top, offset) {
            $('<span role="button" class="button pan-arrow-'+dir+'" style="right:' + right + 'px;top:' + top + 'px"></span>').appendTo($("#zoomNav-"+vitalsGraphDivId)).click(function (e) {
                e.preventDefault();
                plot.pan(offset);
            });
        }

        addArrow('left', 55, 60, { left: -100 });
        addArrow('right', 25, 60, { left: 100 });
        addArrow('up', 40, 45, { top: -100 });
        addArrow('down', 40, 75, { top: 100 });

        // add zoom out button
        $('<div role="button" class="button zoom-buttons" style="top:100px;">zoom in</div>').appendTo($("#zoomNav-"+vitalsGraphDivId)).click(function (e) {
            e.preventDefault();
            plot.zoom();
        });

        var showGraphTooltip = function(x, y, contents) {
            $('<div id="'+vitalsGraphDivId+'_tooltip">' + contents + '</div>').css( {
                position: 'absolute',
                display: 'none',
                'text-align':'center',
                top: y-80,
                left: x-50,
                border: '1px solid #fdd',
                padding: '2px',
                'background-color': '#fee',
                opacity: 0.80
            }).appendTo("body").fadeIn(200);
        }
        var me = this;
        var previousPoint = null;
        $("#"+vitalsGraphDivId).bind("plothover", function (event, pos, item) {

            if (item) {
                if (previousPoint != item.dataIndex) {
                    previousPoint = item.dataIndex;

                    $("#"+vitalsGraphDivId+"_tooltip").remove();
                    var x = item.datapoint[0],
                        y = item.datapoint[1];

                    var records = dataStore.getRange();
                    var idx = item.dataIndex;
                    var record = records[idx];

                    //ensure index matches data

                    //flot graph lib always displays timestamps according to UTC
                    if (record.data.rate == y && Ext.hui.common.createUTCDate(record.data.dateTimeTaken).getTime() == x)
                    {

                        showGraphTooltip(item.pageX, item.pageY,
                                    Ext.hui.common.formatSiteMonikerDisplay(records[idx].data.siteAgency, records[idx].data.siteMoniker) +
                                    "<br/>"+
                                    record.data.rate + " " + record.data.units +
                                    "<br/>" +
                                    Ext.util.Format.dateRenderer('M d, Y H:i')(records[idx].data.dateTimeTaken));
                    }
                }
            }
            else {
                $("#"+vitalsGraphDivId+"_tooltip").remove();
                previousPoint = null;
            }
        });
    }

});
