/*
 * Janus 4.0 (c)
 * Copyright (c) 2011 Hawaii Resource Group LLC. All Rights Reserved.
 * Developed for the Pacific Telehealth & Technology Hui and the Pacific Joint Information Technology Center
 * Contributors:
 *             Honorable Senator Daniel K. Inouye
 *             VA Pacific Islands Health Care System
 *             Tripler Army Medical Center
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with the License.
 *
 * You may obtain a copy of the License at:
 *
 *            http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and limitations under the License.
 */

/**
 * Vitals Window
 * @param {} config Class configuration object
 *
 * @cfg patientToken
 *     Patient's security token
 */
Ext.hui.desktopwindow.patient.Vitals = Ext.extend(Ext.hui.desktopwindow.DesktopWindow,
{
	initComponent: function() {

		this.sortInfo = {field: 'dateTaken', direction: 'DESC'};

        this.bhieSourceReport = null;

        this.dataStore = new Ext.data.GroupingStore({
            restful:true,
            proxy: new Ext.data.HttpProxy({
                url: 'rest/patientVitals.json',
                disableCaching:true,
                timeout: Ext.hui.common.CONNECTION_TIMEOUT_MS,
                method:'GET'
            }),
            baseParams:{
                patToken:this.patientToken,
                startDate:Ext.util.Format.dateRenderer("m/d/y")(this.startDate),
                endDate:Ext.util.Format.dateRenderer("m/d/y")(this.endDate)
            },
            reader: this.createJsonReader(),
            groupField:'dateTaken',
            groupDir:'DESC',
            sortInfo: this.sortInfo,
            listeners:
            {
                exception:{
                    scope:this,
                    fn:function(dataProxy, type, action, options, response, arg){
                        Ext.hui.common.handleNetworkFailure(response.status, this);
                    }
                },
                beforeLoad:
                {
                    scope:this,
                    fn:function()
                    {
                        if (!this || this.isDestroy) return;
                        this.loading = new Ext.LoadMask(this.gridArea.el, {msg:"Loading...", store:this.dataStore});
                        this.loading.show();
                        if (this.filter) this.filter.disable();
                        this.bhieSourceReport = null;
                    }
                },
                load:
                {
                    scope:this,
                    fn:
                    function(store, records, options){
                        if (!this || this.isDestroy) return;
                        if (this.filter)
                        {
                            this.filter.enable();
                            this.filter.loadWithData();
                        }

                    }
                },
                datachanged:
                {
                    scope:this,
                    fn:function(store)
                    {
                        if (!this || this.isDestroy) return;
                        this.tableExpandToggleText = Ext.hui.common.EXPAND_TABLE_TEXT;

                        var recs = store.getRange();
                        var jsonDt = [];
                        var bhieReportRecIdx = -1;
                        for(var i = 0; i < recs.length; i++)
                        {
                            jsonDt[i] = recs[i].json;

                            if (jsonDt[i].siteMoniker == Ext.hui.app.BHIE_STATUS_REPORT_FLAG)
                            {
                                this.bhieSourceReport = jsonDt[i].type;
                                bhieReportRecIdx = i;
                            }
                        }

                        //remove bhie report from list
                        if (bhieReportRecIdx > -1)
                        {
                           jsonDt.remove(jsonDt[bhieReportRecIdx]);
                           store.remove(recs[bhieReportRecIdx]);
                        }

                        this.setTitle("Vitals ("+jsonDt.length+")");

                        var inMemoryStore = new Ext.data.GroupingStore({
                            proxy: new Ext.ux.data.PagingMemoryProxy(jsonDt),
                            remoteSort:true,
                            groupField:'dateTaken',
                            groupDir:'DESC',
                            reader: this.createJsonReader(),
                            sortInfo: this.sortInfo
                        });

                        //destroy previous grid panel
                        if (this.gridPanel)
                        {
                            this.gridPanel.destroy();
                        }
                        this.gridPanel = this.createGrid(inMemoryStore, this, Ext.hui.common.WINDOW_NUM_RECORDS_PER_PAGE);
                        inMemoryStore.load({params:{start:0, limit:Ext.hui.common.WINDOW_NUM_RECORDS_PER_PAGE}});

                        this.gridArea.removeAll();
                        this.gridArea.add(this.gridPanel);
                        this.doLayout();
                        if (this.filter) this.filter.displayFilter(true);
                    }
                }
            },
            autoLoad: true
        });

        this.filter = new Ext.hui.filter.VitalsFilter({
            region:'north',
            dataStore:this.dataStore,
            startDate:this.startDate,
            endDate:this.endDate,
            disabled:true,
            hidden:true
        });

	    this.title='Vitals';
        this.width = 950;
        this.height = 500;
        this.layout = "border";
        this.iconCls = 'icon-grid';
        this.animCollapse = false;
        this.constrainHeader = true;
        this.draggable = true;
        this.dataStore = this.dataStore;
	    this.tools =
	    [{
			id: 'filter',
			qtip: 'Configure Filter',
            cls: 'x-tool-filter',
            overCls:'x-tool-filter-over',
			handler: function (event,toolEl, panel){
				if (panel.filter) {
                    panel.filter.toggleView();
				}
			}
		},
        {
            id:'help',
            qtip: 'Interface Status',
            handler: function(event, toolEl, panel)
            {
                var win = panel.desktop.createWindow({
                    title:"Interface Status ~ Vitals",
                    width: 600,
                    height: 500,
                    autoScroll: true
                });
                win.show();
                var sourceStatus = Ext.hui.app.BHIE_STATUS_NO_REPORT_MSG;

                if (panel.bhieSourceReport != null)
                {
                    sourceStatus = panel.bhieSourceReport;
                }

                win.body.dom.innerHTML = "<div class='bhie-report'>" + sourceStatus + "</div>";
            }
        },
	    {
			id: 'copy',
			qtip: 'Copy to Clipboard',
			handler: function (event,toolEl, panel){
				if (panel.gridPanel) {
					 Ext.hui.common.copyGridPanel(panel.gridPanel);
				}
			}
		},
		{
			id: 'print',
			qtip: 'Print',
			handler: function (event,toolEl, panel){
				Ext.ux.Printer.BaseRenderer.prototype.stylesheetPath = 'css/print.css';
				if (panel.gridPanel) {
					Ext.ux.Printer.printPatInfo(panel.gridPanel);
				}
			}
		},
			{
			    id:'refresh',
			    qtip: 'Refresh',
			    handler: function(event, toolEl, win){
			        win.dataStore.reload();
			    }
			}
		];

	    this.gridArea = new Ext.Container({layout:'fit', region:'center', html:''});
        this.items = [this.filter, this.gridArea];

	    Ext.hui.desktopwindow.patient.Vitals.superclass.initComponent.call(this);
	},
    createJsonReader: function()
    {
        return new Ext.data.JsonReader({
                root: ''
            },
            [
                {name: 'type'},
                {name: 'dateTaken', type:'date'},
                {name: 'result'},
                {name: 'units'},
                {name: 'qualifiers'},
                {name: 'histogramRecordToken'},
                {name: 'siteCode'},
                {name: 'siteMoniker'},
                {name: 'siteAgency'},
                {name: 'sourcePlatform'}
            ]);
    },
    createGrid: function(gridDataStore, refWin, recordsPerPage)
    {

        var pagingToolbar = refWin.createVitalsPagingToolbar(gridDataStore, recordsPerPage);

        var dateRenderer = Ext.util.Format.dateRenderer('M d, Y H:i');

        return new Ext.grid.GridPanel({
            store: gridDataStore,
            columns: [
            {
                header   : 'Site',
                width    : 120,
                sortable : true,
                dataIndex: 'siteMoniker',
                renderer :
                function(value, metaData, record, rowIndex, colIndex, store) {
                    return Ext.hui.common.formatSiteMonikerDisplay(record.data.siteAgency, record.data.siteMoniker);
                }
            },

            {
                width    : 125,
                header   : 'Type',
                sortable : true,
                dataIndex: 'type'
            },
            {
                width    : 80,
                header   : 'Histogram',
                renderer :{
                    fn:function(value, metaData, record, rowIndex, colIndex, store) {
                         if (Ext.hui.common.isVitalGraphAllowed(record.data.type))
                         {
                            return String.format('<span class="histogram-link" id="histogram-graph-link-{0}-{1}">graph</span>&nbsp;|&nbsp;<span class="histogram-link" id="histogram-table-link-{0}-{1}">table</span>', refWin.id, record.id);
                         }
                         else return "Unavailable";
                    },
                    scope:this
                }
            },
            {
                width    : 80,
                header   : 'Result',
                sortable : true,
                dataIndex: 'result'
            },
            {
                width    : 80,
                header   : 'Units',
                sortable : true,
                dataIndex: 'units'
            },
            {
                width    : 120,
                header   : 'Qualifiers',
                sortable : true,
                dataIndex: 'qualifiers'
            },
            {
                width    : 125,
                header   : 'Date Taken',
                sortable : true,
                dataIndex: 'dateTaken',
                renderer: dateRenderer
            }
            ],
            listeners:{
                rowclick:
                {
                    scope:this,
                    fn:function(grid, rowIdx, evtObj) {
                        var record = grid.getStore().getAt(rowIdx);
                        if(evtObj.within(String.format("histogram-graph-link-{0}-{1}", refWin.id, record.id)))
                        {
                            refWin.desktop.createWindow({
                                title:String.format("Vitals Graph ~ {0}",
                                    record.data.type),
                                    histogramRecordToken:record.data.histogramRecordToken,
                                    patientToken:refWin.patientToken
                            }, Ext.hui.desktopwindow.patient.VitalsGraph).show();
                        }
                        else if(evtObj.within(String.format("histogram-table-link-{0}-{1}", refWin.id, record.id)))
                        {
                            refWin.desktop.createWindow({
                                 title:String.format("Vitals Table ~ {0}",
                                    record.data.type),
                                histogramRecordToken:record.data.histogramRecordToken,
                                patientToken:refWin.patientToken,
                                desktop:refWin.desktop
                            }, Ext.hui.desktopwindow.patient.VitalsTable).show();
                        }
                    }
                },
                viewready:
                {
                    scope:refWin,
                    fn:function()
                    {
                        if (!this || this.isDestroy) return;
                        if (this.filter) this.filter.enable();
                        this.loading.hide();
                    }
                }
            },
            view: new Ext.grid.GroupingView({
                       forceFit:true,
                       groupTextTpl: '{text}'
                   }),
            selModel: new Ext.grid.CellSelectionModel(),
            stripeRows: true,
            border:true,
            bodyBorder:true,
            columnLines:true,
            loadMask:true,
            layout:'fit',
            region: 'center',
            bbar: pagingToolbar,
            viewConfig: {
                emptyText:"No results",
                forceFit:true
            }
        });
    },
    createVitalsPagingToolbar: function(dataStore, recordsPerPage, toolbarItems)
    {
        var items = [];
        //do not show expand button if number of records is less than paging amount
        //if in expanded mode, we always want to show collapse button
        var recs = this.dataStore.getRange();
        if (recordsPerPage < recs.length || this.tableExpandToggleText == Ext.hui.common.COLLAPSE_TABLE_TEXT)
        {
            items = [
                    {
                        text: this.tableExpandToggleText,
                        cls: 'more-link',
                        listeners:
                        {
                            scope:this,
                            click: function(btn, pressed){

                                if (!this || this.isDestroy) return;

                                if (this.filter) this.filter.disable();

                                if (this.tableExpandToggleText == Ext.hui.common.EXPAND_TABLE_TEXT)
                                {
                                    this.tableExpandToggleText = Ext.hui.common.COLLAPSE_TABLE_TEXT;
                                }
                                else this.tableExpandToggleText = Ext.hui.common.EXPAND_TABLE_TEXT;

                                var newRecordsPerPage = dataStore.getTotalCount();
                                if (recordsPerPage == newRecordsPerPage)
                                {
                                    newRecordsPerPage = Ext.hui.common.WINDOW_NUM_RECORDS_PER_PAGE;
                                }

                                var recs = this.dataStore.getRange();
                                var jsonDt = [];
                                for(var i = 0; i < recs.length; i++)
                                {
                                    jsonDt[i] = recs[i].json;
                                }

                                var inMemoryStore = new Ext.data.GroupingStore({
                                    proxy: new Ext.ux.data.PagingMemoryProxy(jsonDt),
                                    remoteSort:true,
                                    groupField:'dateTaken',
                                    groupDir:'DESC',
                                    reader: this.createJsonReader(),
                                    sortInfo: this.sortInfo,
                                    listeners:
                                    {
                                        load:
                                        {
                                            scope:this,
                                            fn:function()
                                            {
                                                this.gridArea.add(this.gridPanel);
                                                this.doLayout();
                                            }
                                        }
                                    }
                                });

                                if (newRecordsPerPage == this.dataStore.getTotalCount())
                                {
                                    this.loading.show();
                                }

                                this.gridArea.removeAll();
                                this.gridPanel.destroy();
                                this.gridPanel = this.createGrid(inMemoryStore, this, newRecordsPerPage, this.tableExpandToggleText);

                                inMemoryStore.load({params:{start:0, limit:newRecordsPerPage}});

                            }
                        }
                }];
        }

        if (toolbarItems != null && toolbarItems != undefined)
        {
            items.push(toolbarItems);
        }

        return new Ext.PagingToolbar({
                    pageSize: recordsPerPage,
                    store: dataStore,
                    displayInfo: true,
                    items:items
        });
    }
});

Ext.reg('vitals_window', Ext.hui.desktopwindow.patient.Vitals);