/*
 * Janus 4.0 (c)
 * Copyright (c) 2011 Hawaii Resource Group LLC. All Rights Reserved.
 * Developed for the Pacific Telehealth & Technology Hui and the Pacific Joint Information Technology Center
 * Contributors:
 *             Honorable Senator Daniel K. Inouye
 *             VA Pacific Islands Health Care System
 *             Tripler Army Medical Center
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with the License.
 *
 * You may obtain a copy of the License at:
 *
 *            http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and limitations under the License.
 */

/**
 * Radiology Exams Window
 * @param {} config Class configuration object
 *
 * @cfg patientToken
 *     Patient's security token
 */
Ext.hui.desktopwindow.patient.RadiologyExams = Ext.extend(Ext.hui.desktopwindow.DesktopWindow,
{
	initComponent: function() {

        this.sortInfo = {
            field: 'examDate',
            direction: 'DESC'
        };

        this.dataStore = new Ext.data.Store({
            restful:true,
            proxy: new Ext.data.HttpProxy({
                url: 'rest/patientRadiologyExams.json',
                disableCaching:true,
                timeout: Ext.hui.common.CONNECTION_TIMEOUT_MS,
                method:'GET'
            }),
            baseParams:{
                patToken:this.patientToken,
                startDate:Ext.util.Format.dateRenderer("m/d/y")(this.startDate),
                endDate:Ext.util.Format.dateRenderer("m/d/y")(this.endDate)
            },
            reader: this.createJsonReader(),

            sortInfo: this.sortInfo,
            listeners:
            {
                exception:{
                    scope:this,
                    fn:function(dataProxy, type, action, options, response, arg){
                        Ext.hui.common.handleNetworkFailure(response.status, this);
                    }
                },
                beforeLoad:
                {
                    scope:this,
                    fn:function()
                    {
                        if (!this || this.isDestroy) return;
                        this.loading = new Ext.LoadMask(this.gridArea.el, {msg:"Loading...", store:this.dataStore});
                        this.loading.show();
                        if (this.filter) this.filter.disable();
                    }
                },
                load:
                {
                    scope:this,
                    fn:
                    function(store, records, options){
                        if (!this || this.isDestroy) return;
                        if (this.filter)
                        {
                            this.filter.enable();
                            this.filter.loadWithData();
                        }
                        this.setTitle("Radiology Exams ("+records.length+")");
                    }
                },
                datachanged:
                {
                    scope:this,
                    fn:function(store)
                    {
                        if (!this || this.isDestroy) return;
                        this.tableExpandToggleText = Ext.hui.common.EXPAND_TABLE_TEXT;

                        var recs = store.getRange();
                        var jsonDt = [];
                        for(var i = 0; i < recs.length; i++)
                        {
                            jsonDt[i] = recs[i].json;
                        }

                        var inMemoryStore = new Ext.data.Store({
                            proxy: new Ext.ux.data.PagingMemoryProxy(jsonDt),
                            remoteSort:true,
                            reader: this.createJsonReader(),
                            sortInfo: this.sortInfo
                        });

                        //destroy previous grid panel
                        if (this.gridPanel)
                        {
                            this.gridPanel.destroy();
                        }
                        this.gridPanel = this.createGrid(inMemoryStore, this, Ext.hui.common.WINDOW_NUM_RECORDS_PER_PAGE);
                        inMemoryStore.load({params:{start:0, limit:Ext.hui.common.WINDOW_NUM_RECORDS_PER_PAGE}});

                        this.gridArea.removeAll();
                        this.gridArea.add(this.gridPanel);
                        this.doLayout();
                        if (this.filter) this.filter.displayFilter(true);
                    }
                }
            },
            autoLoad: true
        });

        this.filter = new Ext.hui.filter.RadiologyExamFilter({
            region:'north',
            dataStore:this.dataStore,
            startDate:this.startDate,
            endDate:this.endDate,
            disabled:true,
            hidden:true
        });

	    this.title='Radiology Exams';
        this.width = 950;
        this.height = 500;
        this.layout = "border";
        this.iconCls = 'icon-grid';
        this.animCollapse = false;
        this.constrainHeader = true;
        this.draggable = true;
        this.dataStore = this.dataStore;
	    this.tools =
	    [
        {
			id: 'filter',
			qtip: 'Configure Filter',
            cls: 'x-tool-filter',
            overCls:'x-tool-filter-over',
			handler: function (event,toolEl, panel){
				if (panel.filter) {
                    panel.filter.toggleView();
				}
			}
		},
        {
			id: 'down',
			qtip: 'Download and install Medweb plugin',
			handler: function (event,toolEl, panel){
                window.location.href = Ext.hui.app.MEDWEB_PLUGIN_URL;
            }
		},
        {
			id: 'copy',
			qtip: 'Copy to Clipboard',
            cls: 'x-tool-copy',
            overCls:'x-tool-copy-over',
			handler: function (event,toolEl, panel){
				if (panel.gridPanel) {
                    Ext.hui.common.copyGridPanel(panel.gridPanel);
				}
			}
		},
		{
			id: 'print',
			qtip: 'Print',
			handler: function (event,toolEl, panel){
				Ext.ux.Printer.BaseRenderer.prototype.stylesheetPath = 'css/print.css';
				if (panel.gridPanel) {
					Ext.ux.Printer.printPatInfo(panel.gridPanel);
				}
			}
		},
        {
            id:'refresh',
            qtip: 'Refresh',
            handler: function(event, toolEl, win){
                win.dataStore.reload();
            }
        }
		];

	    this.gridArea = new Ext.Container({layout:'fit', region:'center', html:''});
        this.items = [this.filter, this.gridArea];

	    Ext.hui.desktopwindow.patient.RadiologyExams.superclass.initComponent.call(this);
	},
    createJsonReader: function()
    {
        return new Ext.data.JsonReader({
                root: ''
            },
            [
                {name: 'study'},
                {name: 'examDate', type:'date'},
                {name: 'status'},
                {name: 'medWebAvailable'},
                {name: 'medWebToken'},
                {name: 'recordDetailsToken'},
                {name: 'siteCode'},
                {name: 'siteMoniker'},
                {name: 'siteAgency'},
                {name: 'sourcePlatform'}
            ]);
    },
    createGrid: function(gridDataStore, refWin, recordsPerPage)
    {
        var dateRenderer = Ext.util.Format.dateRenderer('M d, Y H:i');

        return new Ext.grid.GridPanel({
                store: gridDataStore,
                columns: [
                    {
                        header   : 'Site',
                        width    : 60,
                        sortable : true,
                        dataIndex: 'siteMoniker',
                        renderer :
                        function(value, metaData, record, rowIndex, colIndex, store) {
                            return Ext.hui.common.formatSiteMonikerDisplay(record.data.siteAgency, record.data.siteMoniker);
                        }
                    },
                    {
                        width    : 150,
                        header   : 'Date',
                        sortable : true,
                        dataIndex: 'examDate',
                        renderer: dateRenderer
                    },
                    {
                        id: 'study',
                        width    : 200,
                        header   : 'Study',
                        sortable : true,
                        dataIndex: 'study',
                        renderer :{
                            fn:function(value, metaData, record, rowIndex, colIndex, store) {
                                if (record.data.recordDetailsToken != null)
                                {
                                    return String.format('<span class="record-details" id="radiologyreport-details-{0}-{1}">{2}</span>', refWin.id, record.id, record.data.study);
                                }
                                else return record.data.study;

                            },
                            scope:this
                        }
                    },
                    {
                        id:'status',
                        width:120,
                        header: 'Status',
                        sortable:true,
                        dataIndex:'status'

                    },
                    {
                        width:35,
                        header: 'Image',
                        id:'image',
                        sortable: 'false',
                        renderer:{
                            fn:function(value, metaData, record, rowIndex, colIndex, store) {
                                if (record.data.recordDetailsToken != null)
                                {
                                    if(!record.data.medWebAvailable)
                                    {
                                        return "";
                                    }

                                    var medWebURL = String.format("patientMedWeb?patToken={0}&recToken={1}",
                                            refWin.patientToken, record.data.medWebToken);

                                    var medWebCmd = String.format("Ext.hui.common.openAuxBrowserWindow(\"{0}\", \"IEHR_MEDWEB\");return false;", medWebURL);

                                    return String.format('<span class="medweb-link" title="View in MedWeb" alt="View in MedWeb" onclick=\'{0}\'><img src="images/camera_icon.png"  alt=\'Camera Icon\'/></span>', medWebCmd);
                                }
                                return "";

                            },
                            scope:this
                        }
                    }
                ],
                listeners:{
                    rowclick:
                    {
                        scope:this,
                        fn:function(grid, rowIdx, evtObj) {
                            var record = grid.getStore().getAt(rowIdx);
                            if(evtObj.within(String.format("radiologyreport-details-{0}-{1}", refWin.id, record.id)))
                            {
                                refWin.desktop.createWindow({
                                    title:String.format("Radiology Report ~ {0} {1}",
                                            Ext.hui.common.formatSiteMonikerDisplay(record.data.siteAgency, record.data.siteMoniker),
                                            record.data.study),
                                    patientToken:refWin.patientToken,
                                    desktop:refWin.desktop,
                                    recordDetailsToken:record.data.recordDetailsToken
                                }, Ext.hui.desktopwindow.patient.RadiologyReport).show();
                            }
                        }
                    },
                    viewready:
                    {
                        scope:refWin,
                        fn:function()
                        {
                            if (!this || this.isDestroy) return;
                            if (this.filter) this.filter.enable();
                            this.loading.hide();
                        }
                    }
                },
            selModel: new Ext.grid.CellSelectionModel(),
                stripeRows: true,
                autoExpandColumn:'study',
                border:true,
                bodyBorder:true,
                loadMask:true,
                layout:'fit',
                region: 'center',
                columnLines:true,
                bbar: refWin.createPagingToolbar(gridDataStore, recordsPerPage),
                viewConfig: {
                    emptyText:"No results",
                    forceFit:true
                }
        });
    }
});
Ext.reg('radiologyexams_window', Ext.hui.desktopwindow.patient.RadiologyExams);