/*
 * Janus 4.0 (c)
 * Copyright (c) 2011 Hawaii Resource Group LLC. All Rights Reserved.
 * Developed for the Pacific Telehealth & Technology Hui and the Pacific Joint Information Technology Center
 * Contributors:
 *             Honorable Senator Daniel K. Inouye
 *             VA Pacific Islands Health Care System
 *             Tripler Army Medical Center
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with the License.
 *
 * You may obtain a copy of the License at:
 *
 *            http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and limitations under the License.
 */

/**
 * Progress Notes Window
 * @param {} config Class configuration object
 *
 * @cfg patientToken
 *     Patient's security token
 */
Ext.hui.desktopwindow.patient.ProgressNotes = Ext.extend(Ext.hui.desktopwindow.DesktopWindow,
{
	initComponent: function() {

        this.toolsCfg = [
            {
                id: 'filter',
                qtip: 'Configure Filter',
                cls: 'x-tool-filter',
                overCls:'x-tool-filter-over',
                handler: function (event,toolEl, panel){
                    if (panel.filter) {
                        panel.filter.toggleView();
                    }
                }
            },
            {
                id:'help',
                qtip: 'Interface Status',
                handler: function(event, toolEl, panel)
                {
                     var win = panel.desktop.createWindow({
                        title:"Interface Status ~ Progress Notes",
                        width: 600,
                        height: 500,
                        autoScroll: true
                    });
                    win.show();
                    var sourceStatus = Ext.hui.app.BHIE_STATUS_NO_REPORT_MSG;

                    if (panel.bhieSourceReport != null)
                    {
                        sourceStatus = panel.bhieSourceReport;
                    }

                    win.body.dom.innerHTML = "<div class='bhie-report'>" + sourceStatus + "</div>";
                }
            },
            {
                id: 'copy',
                qtip: 'Copy to Clipboard',
                cls: 'x-tool-copy',
                overCls:'x-tool-copy-over',
                handler: function (event,toolEl, panel){
                    if (panel.gridPanel) {
                        Ext.hui.common.copyGridPanel(panel.gridPanel);
                    }
                }
            },
            {
                id: 'print',
                qtip: 'Print',
                handler: function (event,toolEl, panel){
                    Ext.ux.Printer.BaseRenderer.prototype.stylesheetPath = 'css/print.css';
                    if (panel.gridPanel) {
                        Ext.ux.Printer.printPatInfo(panel.gridPanel);
                    }
                }
            },
            {
                id:'refresh',
                qtip: 'Refresh',
                handler: function(event, toolEl, win){
                    win.dataStore.reload();
                }
            }
        ];

        if (Ext.hui.app.WRITEBACK_ENABLE=="true"){
            this.toolsCfg.splice(0,0,
                {
                    id: 'addNote',
                    qtip: 'New Note',
                    cls: 'x-tool-addNote',
                    overCls:'x-tool-addNote-over',
                    handler: function (event,toolEl, panel){
                        panel.desktop.createWindow({
                            title:String.format("New Note"),
                            patientToken:panel.patientToken,
                            desktop:panel.desktop
                        }, Ext.hui.desktopwindow.patient.AddNote).show();
                    }
                }
            )
        };

        this.sortInfo = {
            field: 'noteDate',
            direction: 'DESC'
        };

        this.bhieSourceReport = null;

        this.dataStore = new Ext.data.Store({
            restful:true,
            proxy: new Ext.data.HttpProxy({
                url: 'rest/patientProgressNotes.json',
                disableCaching:true,
                timeout: Ext.hui.common.CONNECTION_TIMEOUT_MS,
                method:'GET'
            }),
            baseParams:{
                patToken:this.patientToken,
                startDate:Ext.util.Format.dateRenderer("m/d/y")(this.startDate),
                endDate:Ext.util.Format.dateRenderer("m/d/y")(this.endDate)
            },
            reader: this.createJsonReader(),

            sortInfo: this.sortInfo,
            listeners:
            {
                exception:{
                    scope:this,
                    fn:function(dataProxy, type, action, options, response, arg){
                        Ext.hui.common.handleNetworkFailure(response.status, this);
                    }
                },
                beforeLoad:
                {
                    scope:this,
                    fn:function()
                    {
                        if (!this || this.isDestroy) return;
                        this.loading = new Ext.LoadMask(this.gridArea.el, {msg:"Loading...", store:this.dataStore});
                        this.loading.show();
                        if (this.filter) this.filter.disable();
                        this.bhieSourceReport = null;
                    }
                },
                load:
                {
                    scope:this,
                    fn:
                    function(store, records, options){
                        if (!this || this.isDestroy) return;
                        if (this.filter)
                        {
                            this.filter.enable();
                            this.filter.loadWithData();
                        }
                    }
                },
                datachanged:
                {
                    scope:this,
                    fn:function(store)
                    {
                        if (!this || this.isDestroy) return;
                        this.tableExpandToggleText = Ext.hui.common.EXPAND_TABLE_TEXT;

                        var recs = store.getRange();
                        var jsonDt = [];
                        var bhieReportRecIdx = -1;
                        for(var i = 0; i < recs.length; i++)
                        {
                            jsonDt[i] = recs[i].json;

                            if (jsonDt[i].siteMoniker == Ext.hui.app.BHIE_STATUS_REPORT_FLAG)
                            {
                                this.bhieSourceReport = jsonDt[i].noteType;
                                bhieReportRecIdx = i;
                            }
                        }

                        //remove bhie report from list
                        if (bhieReportRecIdx > -1)
                        {
                           jsonDt.remove(jsonDt[bhieReportRecIdx]);
                           store.remove(recs[bhieReportRecIdx]);
                        }

                        this.setTitle("Progress Notes ("+jsonDt.length+")");

                        var inMemoryStore = new Ext.data.Store({
                            proxy: new Ext.ux.data.PagingMemoryProxy(jsonDt),
                            remoteSort:true,
                            reader: this.createJsonReader(),
                            sortInfo: this.sortInfo
                        });

                        //destroy previous grid panel
                        if (this.gridPanel)
                        {
                            this.gridPanel.destroy();
                        }
                        this.gridPanel = this.createGrid(inMemoryStore, this, Ext.hui.common.WINDOW_NUM_RECORDS_PER_PAGE);
                        inMemoryStore.load({params:{start:0, limit:Ext.hui.common.WINDOW_NUM_RECORDS_PER_PAGE}});

                        this.gridArea.removeAll();
                        this.gridArea.add(this.gridPanel);
                        this.doLayout();
                        if (this.filter) this.filter.displayFilter(true);
                    }
                }
            },
            autoLoad: true
        });

        this.filter = new Ext.hui.filter.NoteFilter({
            region:'north',
            dataStore:this.dataStore,
            startDate:this.startDate,
            endDate:this.endDate,
            disabled:true,
            hidden:true
        });

	    this.title='Progress Notes';
        this.width = 950;
        this.height = 500;
        this.layout = "border";
        this.iconCls = 'icon-grid';
        this.animCollapse = false;
        this.constrainHeader = true;
        this.draggable = true;
        this.dataStore = this.dataStore;
	    this.tools = this.toolsCfg;

        this.gridArea = new Ext.Container({layout:'fit', region:'center', html:''});
	    this.items = [this.filter, this.gridArea];

	    Ext.hui.desktopwindow.patient.ProgressNotes.superclass.initComponent.call(this);
	},
    createJsonReader: function()
    {
        return new Ext.data.JsonReader({
                root: ''
            },
            [
                {name: 'noteType'},
                {name: 'noteDate', type:'date'},
                {name: 'location'},
                {name: 'status'},
                {name: 'providerName'},
                {name: 'visitAdm'},
                {name: 'recordDetailsToken'},
                {name: 'nonPlainTextNote'},
                {name: 'siteCode'},
                {name: 'siteMoniker'},
                {name: 'siteAgency'},
                {name: 'sourcePlatform'}
            ]);
    },
    createGrid: function(gridDataStore, refWin, recordsPerPage)
    {
        var pagingToolbar = refWin.createPagingToolbar(gridDataStore, recordsPerPage);

        var dateRenderer = Ext.util.Format.dateRenderer('M d, Y H:i');

        return new Ext.grid.GridPanel({
                store: gridDataStore,
                columns: [
                {
                    header   : 'Site',
                    width    : 120,
                    sortable : true,
                    dataIndex: 'siteMoniker',
                    renderer :
                    function(value, metaData, record, rowIndex, colIndex, store) {
                        return Ext.hui.common.formatSiteMonikerDisplay(record.data.siteAgency, record.data.siteMoniker);
                    }
                },
                {
                    width    : 120,
                    header   : 'Date',
                    sortable : true,
                    dataIndex: 'noteDate',
                    renderer: dateRenderer
                },
                {
                    id: 'noteType',
                    width    : 200,
                    header   : 'Type',
                    sortable : true,
                    dataIndex: 'noteType',
                    renderer :{
                        fn:function(value, metaData, record, rowIndex, colIndex, store) {
                            if (record.data.recordDetailsToken != null)
                            {
                                var noteType = record.data.noteType;
                                if (noteType.length < 1)
                                {
                                    noteType = "Note type unavailable, click link to view note";
                                }

                                return String.format('<span class="record-details" id="progressnote-details-{0}-{1}">{2}</span>', refWin.id, record.id, noteType);
                            }
                            else return record.data.noteType;

                        },
                        scope:this
                    }
                },
                {
                    width    : 200,
                    header   : 'Provider',
                    sortable : true,
                    dataIndex: 'providerName'
                },
                {
                    width    : 100,
                    header   : 'Location',
                    sortable : true,
                    dataIndex: 'location'
                },
                {
                    width    : 100,
                    header   : 'Visit/Adm Date',
                    sortable : true,
                    dataIndex: 'visitAdm'
                },
                {
                    width    : 100,
                    header   : 'Status',
                    sortable : true,
                    dataIndex: 'status'
                }
                ],
                listeners:{
                    rowclick:
                    {
                        scope:this,
                        fn:function(grid, rowIdx, evtObj) {
                            var record = grid.getStore().getAt(rowIdx);
                            if(evtObj.within(String.format("progressnote-details-{0}-{1}", refWin.id, record.id)))
                            {
                                if (record.data.nonPlainTextNote)
                                {
                                    var noteURL = String.format("rest/patientProgressNoteDetails?patToken={0}&recToken={1}",
                                            refWin.patientToken, record.data.recordDetailsToken);

                                    Ext.hui.common.openAuxBrowserWindow(noteURL, "IEHR_PROGRESS_NOTE");
                                }
                                else
                                {
                                    refWin.desktop.createWindow({
                                        title:String.format("Progress Note Details ~ {0} {1}",
                                                Ext.hui.common.formatSiteMonikerDisplay(record.data.siteAgency, record.data.siteMoniker),
                                                record.data.noteType),
                                        patientToken:refWin.patientToken,
                                        desktop:refWin.desktop,
                                        recordDetailsToken:record.data.recordDetailsToken
                                    }, Ext.hui.desktopwindow.patient.ProgressNoteDetails).show();
                                }
                            }
                        }
                    },
                    viewready:
                    {
                        scope:refWin,
                        fn:function()
                        {
                            if (!this || this.isDestroy) return;
                            if (this.filter) this.filter.enable();
                            this.loading.hide();

                        }
                    }
                },
            selModel: new Ext.grid.CellSelectionModel(),
                stripeRows: true,
                autoExpandColumn:'noteType',
                border:true,
                bodyBorder:true,
                loadMask:true,
                layout:'fit',
                region: 'center',
                columnLines:true,
                bbar: pagingToolbar,
                viewConfig: {
                    emptyText:"No results",
                    forceFit:true
                }
        });

    }
});
//Ext.reg('progressnotes_window', Ext.hui.desktopwindow.patient.ProgressNotes);

//SB: Using old alias to support widget creation from save data
Ext.reg('outpatientnotes_window', Ext.hui.desktopwindow.patient.ProgressNotes);