/*
 * Janus 4.0 (c)
 * Copyright (c) 2011 Hawaii Resource Group LLC. All Rights Reserved.
 * Developed for the Pacific Telehealth & Technology Hui and the Pacific Joint Information Technology Center
 * Contributors:
 *             Honorable Senator Daniel K. Inouye
 *             VA Pacific Islands Health Care System
 *             Tripler Army Medical Center
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with the License.
 *
 * You may obtain a copy of the License at:
 *
 *            http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and limitations under the License.
 */

/**
 * Patient Problem List Window
 * @param {} config Class configuration object
 *
 * @cfg patientToken
 *     Patient's security token
 */
Ext.hui.desktopwindow.patient.ProblemList = Ext.extend(Ext.hui.desktopwindow.DesktopWindow,
    {
        initComponent: function() {

            this.sortInfo = {
                field: 'lastUpdated',
                direction: 'DESC'
            };

            this.bhieSourceReport = null;

            this.dataStore = new Ext.data.Store({
                restful:true,
                proxy: new Ext.data.HttpProxy({
                    /** Always grab all problems, status filtering performed on grid **/
                    url: 'rest/patientProblemList.json',
                    disableCaching:true,
                    timeout: Ext.hui.common.CONNECTION_TIMEOUT_MS,
                    method:'GET'
                }),
                baseParams:
                {
                    patToken:this.patientToken,
                    status: 'A'
                },
                reader: this.createJsonReader(),

                sortInfo: this.sortInfo,
                listeners:
                {
                    exception:{scope:this, fn:function(dataProxy, type, action, options, response, arg) {
                        this.dataStore.data.items = null;
                        Ext.hui.common.handleNetworkFailure(response.status, this);
                    }},
                    beforeLoad:
                    {
                        scope:this,
                        fn:function() {
                            if (!this || this.isDestroy) return;
                            this.loading = new Ext.LoadMask(this.gridArea.el, {msg:"Loading...", store:this.dataStore});
                            this.loading.show();
                            this.bhieSourceReport = null;
                        }
                    },
                    load:
                    {
                        scope:this,
                        fn:
                            function(store, records, options) {

                            }
                    },
                    datachanged:
                    {
                        scope:this,
                        fn:function(store) {
                            if (!this || this.isDestroy) return;

                            this.tableExpandToggleText = Ext.hui.common.EXPAND_TABLE_TEXT;

                            var recs = store.getRange();
                            var jsonDt = [];
                            var bhieReportRecIdx = -1;
                            for (var i = 0; i < recs.length; i++) {
                                jsonDt[i] = recs[i].json;

                                if (jsonDt[i].siteMoniker == Ext.hui.app.BHIE_STATUS_REPORT_FLAG)
                                {
                                    this.bhieSourceReport = jsonDt[i].description;
                                    bhieReportRecIdx = i;
                                }
                            }

                            //remove bhie report from list
                            if (bhieReportRecIdx > -1)
                            {
                               jsonDt.remove(jsonDt[bhieReportRecIdx]);
                               store.remove(recs[bhieReportRecIdx]);
                            }

                            this.setTitle("Problem List (" + jsonDt.length + ")");

                            var inMemoryStore = new Ext.data.Store({
                                proxy: new Ext.ux.data.PagingMemoryProxy(jsonDt),
                                remoteSort:true,
                                reader: this.createJsonReader(),
                                sortInfo: this.sortInfo
                            });

                            //destroy previous grid panel
                            if (this.gridPanel) {
                                this.gridPanel.destroy();
                            }
                            this.gridPanel = this.createGrid(inMemoryStore, this, Ext.hui.common.WINDOW_NUM_RECORDS_PER_PAGE);
                            inMemoryStore.load({params:{start:0, limit:Ext.hui.common.WINDOW_NUM_RECORDS_PER_PAGE}});

                            this.gridArea.removeAll();
                            this.gridArea.add(this.gridPanel);
                            this.doLayout();
                        }
                    }
                },
                autoLoad: true
            });

            this.title = 'Problem List';
            this.width = 780;
            this.height = 450;
            this.iconCls = 'icon-grid';
            this.animCollapse = false;
            this.constrainHeader = true;
            this.draggable = true;
            this.layout = 'fit';
            this.dataStore = this.dataStore;
            this.tools =
                [
                    {
                        id:'help',
                        qtip: 'Interface Status',
                        handler: function(event, toolEl, panel)
                        {
                            var win = panel.desktop.createWindow({
                                title:"Interface Status ~ Problem List",
                                width: 600,
                                height: 500,
                                autoScroll: true
                            });
                            win.show();
                            var sourceStatus = Ext.hui.app.BHIE_STATUS_NO_REPORT_MSG;

                            if (panel.bhieSourceReport != null)
                            {
                                sourceStatus = panel.bhieSourceReport;
                            }

                            win.body.dom.innerHTML = "<div class='bhie-report'>" + sourceStatus + "</div>";
                        }
                    },
                    {
                        id: 'copy',
                        qtip: 'Copy to Clipboard',
                        handler: function (event, toolEl, panel) {
                            if (panel.gridPanel) {
                                Ext.hui.common.copyGridPanel(panel.gridPanel);
                            }
                        }
                    },

                    {
                        id: 'print',
                        qtip: 'Print',
                        handler: function (event, toolEl, panel) {
                            Ext.ux.Printer.BaseRenderer.prototype.stylesheetPath = 'css/print.css';
                            if (panel.gridPanel) {
                                Ext.ux.Printer.printPatInfo(panel.gridPanel);
                            }
                        }
                    },
                    {
                        id:'refresh',
                        qtip: 'Refresh',
                        handler: function(event, toolEl, win) {
                            win.dataStore.reload();
                        }
                    }
                ];

            this.gridArea = new Ext.Container({layout:'fit', region:'center', html:''});
            this.items = [this.gridArea];

            Ext.hui.desktopwindow.patient.ProblemList.superclass.initComponent.call(this);
        },
        createJsonReader: function() {
            return new Ext.data.JsonReader({
                    root: ''
                },
                [
                    {name: 'onsetDate', type:'date'},
                    {name: 'icdCode'},
                    {name: 'status'},
                    {name: 'lastUpdated', type:'date'},
                    {name: 'description'},
                    {name: 'severity', mapping: 'priority'},
                    {name: 'recordDetailsToken'},
                    {name: 'siteCode'},
                    {name: 'siteMoniker'},
                    {name: 'siteAgency'},
                    {name: 'sourcePlatform'}
                ]);
        },
        createGrid: function(gridDataStore, refWin, recordsPerPage) {

            var pagingToolbar = refWin.createPagingToolbar(gridDataStore, recordsPerPage);

            var dateRenderer = Ext.util.Format.dateRenderer('M d, Y');

            return new Ext.grid.GridPanel({
                store: gridDataStore,
                columns: [
                    {
                        header   : 'Site',
                        width    : 120,
                        sortable : true,
                        dataIndex: 'siteMoniker',
                        renderer :
                            function(value, metaData, record, rowIndex, colIndex, store) {
                                return Ext.hui.common.formatSiteMonikerDisplay(record.data.siteAgency, record.data.siteMoniker);
                            }
                    },
                    {
                        width    : 100,
                        header   : 'Onset Date',
                        sortable : true,
                        dataIndex: 'onsetDate',
                        renderer: dateRenderer
                    },
                    {
                        width    : 100,
                        header   : 'Updated',
                        sortable : true,
                        dataIndex: 'lastUpdated',
                        renderer: dateRenderer
                    },
                    {
                        id: 'description',
                        width    : 130,
                        header   : 'Description',
                        sortable : true,
                        dataIndex: 'description',
                        renderer :{
                            fn:function(value, metaData, record, rowIndex, colIndex, store) {
                                if (record.data.recordDetailsToken != null) {
                                    return String.format('<span class="record-details" id="problem-details-{0}-{1}">{2}</span>', refWin.id, record.id, record.data.description);
                                }
                                else return record.data.description;

                            },
                            scope:this
                        }
                    },
                    {
                        header   : 'ICD Code',
                        width    : 100,
                        sortable : true,
                        dataIndex: 'icdCode'
                    },
                    {
                        header   : 'Severity',
                        width    : 75,
                        sortable : true,
                        dataIndex: 'severity'
                    },
                    {
                        header   : 'Status',
                        width    : 75,
                        sortable : true,
                        dataIndex: 'status'
                    }
                ],
                listeners:{
                    rowclick:
                    {
                        scope:this,
                        fn:function(grid, rowIdx, evtObj) {
                            var record = grid.getStore().getAt(rowIdx);
                            if (evtObj.within(String.format("problem-details-{0}-{1}", refWin.id, record.id))) {
                                refWin.desktop.createWindow({
                                    title:String.format("Problem Details ~ {0} {1}",
                                        Ext.hui.common.formatSiteMonikerDisplay(record.data.siteAgency, record.data.siteMoniker),
                                        record.data.description),
                                    patientToken:refWin.patientToken,
                                    desktop:refWin.desktop,
                                    recordDetailsToken:record.data.recordDetailsToken
                                }, Ext.hui.desktopwindow.patient.ProblemDetails).show();
                            }
                        }
                    },
                    viewready:
                    {
                        scope:refWin,
                        fn:function() {
                            if (!this || this.isDestroy) return;
                            this.loading.hide();
                        }
                    }
                },
                selModel: new Ext.grid.CellSelectionModel(),
                stripeRows: true,
                columnLines:true,
                autoExpandColumn:'description',
                loadMask:true,
                bbar: pagingToolbar,
                viewConfig: {
                    emptyText:"No results",
                    forceFit:true
                },
                cls:"problemlist"
            });
        }
    });

Ext.reg('problemlist_window', Ext.hui.desktopwindow.patient.ProblemList);