/*
 * Janus 4.0 (c)
 * Copyright (c) 2011 Hawaii Resource Group LLC. All Rights Reserved.
 * Developed for the Pacific Telehealth & Technology Hui and the Pacific Joint Information Technology Center
 * Contributors:
 *             Honorable Senator Daniel K. Inouye
 *             VA Pacific Islands Health Care System
 *             Tripler Army Medical Center
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with the License.
 *
 * You may obtain a copy of the License at:
 *
 *            http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and limitations under the License.
 */

/**
 * Patient Search Desktop Window
 * @param {} config Class configuration object
 * 
 * @cls searchQuery
 * 	Patient search paramters
 */
Ext.hui.desktopwindow.patient.PatientSearch = Ext.extend(Ext.hui.desktopwindow.DesktopWindow,
{
	initComponent: function() {

        this.sortInfo = {
            field: 'name',
            direction: 'ASC'
        };

        this.dataStore = new Ext.data.Store({
			restful:true,
			proxy: new Ext.data.HttpProxy({
						url: 'rest/patientSearch.json',
						disableCaching:true,
						method:'GET',
                        timeout: Ext.hui.common.CONNECTION_TIMEOUT_MS
	        		}),
            baseParams:
            {
                query:this.searchQuery
            },
	        reader: this.createJsonReader(),
	        listeners:
	        {
	        	exception:{scope:this, fn:
	        			function(dataProxy, type, action, options, response, arg)
						{	
							Ext.hui.common.handleNetworkFailure(response.status, this);
						}
	        	},
                beforeLoad:
                {
                    scope:this,
                    fn:function()
                    {
                        if (!this || this.isDestroy) return;
                        this.loading = new Ext.LoadMask(this.gridArea.el, {msg:"Loading...", store:this.dataStore});
                        this.loading.show();
                    }
                },
                datachanged:
                {
                    scope:this,
                    fn:function(store)
                    {
                        if (!this || this.isDestroy) return;
                        this.tableExpandToggleText = Ext.hui.common.EXPAND_TABLE_TEXT;

                        var recs = store.getRange();
                        var jsonDt = [];
                        for(var i = 0; i < recs.length; i++)
                        {
                            jsonDt[i] = recs[i].json;
                        }

                        var inMemoryStore = new Ext.data.Store({
                            proxy: new Ext.ux.data.PagingMemoryProxy(jsonDt),
                            remoteSort:true,
                            reader: this.createJsonReader(),
                            sortInfo: this.sortInfo
                        });

                        //destroy previous grid panel
                        if (this.gridPanel)
                        {
                            this.gridPanel.destroy();
                        }
                        this.gridPanel = this.createGrid(inMemoryStore, this, Ext.hui.common.WINDOW_NUM_RECORDS_PER_PAGE);
                        inMemoryStore.load({params:{start:0, limit:Ext.hui.common.WINDOW_NUM_RECORDS_PER_PAGE}});

                        this.gridArea.removeAll();
                        this.gridArea.add(this.gridPanel);
                        this.doLayout();

                    }
                }
	        },
            sortInfo: this.sortInfo,
	        autoLoad: true
	    });
	    
	    this.title='Patient Search - '+this.searchQuery;
        this.width = 720;
        this.height = 450;
        this.iconCls = 'icon-search';
        this.animCollapse = false;
        this.constrainHeader = true;
        this.draggable = true;
        this.layout = 'fit';

        this.gridArea = new Ext.Container({layout:'fit', region:'center', html:''});

        this.items = [this.gridArea];
	    
		Ext.hui.desktopwindow.patient.PatientSearch.superclass.initComponent.call(this);
	},
    createJsonReader: function()
    {
        return new Ext.data.JsonReader({
	            root: ''
	        },
	        [
				{name: 'name'},
				{name: 'age'},
				{name: 'gender'},
				{name: 'fmpssn'},
                {name: 'sensitive'},
				{name: 'patientToken'}
	        ]);
    },
    createGrid: function(gridDataStore, refWin, recordsPerPage)
    {
         return new Ext.grid.GridPanel({
	        store: gridDataStore,
	        columns: [
	            {
	                id       :'name',
	                header   : 'Name',
	                width    : 160,
	                sortable : true,
	                dataIndex:'name',
	                renderer :
	                        function(value, metaData, record, rowIndex, colIndex, store) {
	                            return String.format('<span id="patientsearch-link-{0}-{1}" class="gridLink">'+record.data.name+'</span>', refWin.id, record.id);
	                        }
	            },
	            {
	                header   : 'FMPSSN',
	                width    : 100,
	                sortable : true,
	                dataIndex: 'fmpssn'
	            },
	            {
	                header   : 'Age/Gender',
	                width    : 75,
	                sortable : true,
	                renderer :
	                        function(value, metaData, record, rowIndex, colIndex, store) {
	                            return record.data.age + "/" + record.data.gender;
	                        }
	            }
	        ],
            listeners:{
	        	rowclick:
                {
                    scope:this,
                    fn:function(grid, rowIdx, evtObj) {
                        var record = grid.getStore().getAt(rowIdx);
                        if(evtObj.within(String.format("patientsearch-link-{0}-{1}", refWin.id, record.id)))
                        {
                            Ext.hui.app.selectPatientFromPatientSearch(record.data.patientToken, record.data.sensitive, record.data.name, refWin.id);
                        }
                    }
                },
                viewready:
                {
                    scope:refWin,
                    fn:function()
                    {
                        if (!this || this.isDestroy) return;
                        this.loading.hide();
                    }
                }
	        },
            selModel: new Ext.grid.CellSelectionModel(),
	        stripeRows: true,
	        autoExpandColumn:'name',
	        loadMask:true,
            columnLines:true,
            bbar: refWin.createPagingToolbar(gridDataStore, recordsPerPage),
	        viewConfig: {
	        	emptyText:"No results",
                forceFit:true
	        },
	        cls:"patientsearch"
	    });
    }
	
});

Ext.reg('patientsearch_window', Ext.hui.desktopwindow.patient.PatientSearch);
