/*
 * Janus 4.0 (c)
 * Copyright (c) 2011 Hawaii Resource Group LLC. All Rights Reserved.
 * Developed for the Pacific Telehealth & Technology Hui and the Pacific Joint Information Technology Center
 * Contributors:
 *             Honorable Senator Daniel K. Inouye
 *             VA Pacific Islands Health Care System
 *             Tripler Army Medical Center
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with the License.
 *
 * You may obtain a copy of the License at:
 *
 *            http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and limitations under the License.
 */

/**
 * Lab Table Window
 * @param {} config Class configuration object
 *
 */
Ext.hui.desktopwindow.patient.LabTable = Ext.extend(Ext.hui.desktopwindow.DesktopWindow,
{
    initComponent: function() {
		if (!this.title || this.title.length < 0)
        {
            this.title = "Lab Table";
        }
	    this.width = 800;
        this.height = 510;
        this.cls = "labtable";
        this.autoScroll = true;

        this.on('afterrender', function(){this.fetchData()}, this);

        this.addTool({
            id: 'copy',
			qtip: 'Copy to Clipboard',
            cls: 'x-tool-copy',
            overCls:'x-tool-copy-over',
			handler: function (event,toolEl, window){
				if (window) {
                    Ext.hui.common.copyWindow(window);
				}
			}
        });
        this.addTool({
            id: 'print',
			qtip: 'Print',
			handler: function (event,toolEl, window){
				Ext.ux.Printer.BaseRenderer.prototype.stylesheetPath = 'css/print.css';
				if (window) {
					Ext.ux.Printer.printPatInfo(window);
				}
			}
        });
	    Ext.hui.desktopwindow.patient.LabTable.superclass.initComponent.call(this);
	},

	fetchData: function()
	{
		this.getUpdater().setRenderer(new Ext.hui.desktopwindow.patient.LabTableRenderer(this));
		this.load({
			url: 'rest/patientLabResultHistogram.json',
			method:'GET',
			params: {
                patToken:this.labResult.patientToken,
                recToken:this.labResult.histogramRecordToken
            },
			discardUrl: true,
		    nocache: true,
            disableCaching:true,
		    text: 'Loading...',
		    timeout:Ext.hui.common.CONNECTION_TIMEOUT_SECONDS,
		    scripts: false,
		    scope:this,
	    	callback:function(options, isSuccess, response)
			{
				if (!isSuccess)
				{
					Ext.hui.common.handleNetworkFailure(response.status, this);
				}
			}
		});
	}

});

Ext.reg('labtable_window', Ext.hui.desktopwindow.patient.LabTable);

/**
 * Lab table renderer
 *
 * @param window The window to render into.
 *
 */
Ext.hui.desktopwindow.patient.LabTableRenderer = function(window)
{
	this.window = window;
	Ext.hui.desktopwindow.patient.LabTableRenderer.superclass.constructor.call(this);
}

Ext.extend(Ext.hui.desktopwindow.patient.LabTableRenderer, Ext.Updater.BasicRenderer,
{
    render: function(el, xhr, updateManager, callback)
	{
        if (!this.window || this.window.isDestroy) return;

        var jsonResponse = Ext.util.JSON.decode(xhr.responseText);

        var generateData = function(records)
        {
            var data = [];

            for(var i = 0; i < records.length; i++)
            {
                var labResult = records[i];

                var bgColor = "white";
                var abnFlag = false;
                if (labResult.data.hiLoFlag && labResult.data.hiLoFlag.length > 0)
                {
                    bgColor="#FFCCCC";
                    if (labResult.data.hiLoFlag.indexOf("*") > 0)
                    {
                        bgColor = "red";
                        abnFlag = true;
                    }
                }

                data.push
                ({
                    'date':Ext.util.Format.dateRenderer('M d, Y H:i')(labResult.data.resultDate),
                    'result':parseFloat(labResult.data.result),
                    'specimen':labResult.data.specimen,
                    'units':labResult.data.units,
                    'testName':labResult.data.testName,
                    'hiLoFlag':labResult.data.hiLoFlag,
                    'referenceRange':labResult.data.referenceRange,
                    'comment':"Comment:\n"+labResult.data.comment,
                    'abnFlag':abnFlag,
                    'bgColor':bgColor
                });
            }

            return data;
        }


		el.dom.innerHTML = "";

		if (jsonResponse.guiLabResults == null || jsonResponse.guiLabResults.length < 1)
        {
            this.window.body.dom.innerHTML = '<div class="errortext">An error occurred when attempting to retrieve table data for this lab result series.</div>';
        }
        else
        {
            var labTableDivId = "labtable_" + this.window.id;
            var firstResult = jsonResponse.guiLabResults[0];
            jsonResponse.labTitle = firstResult.testName;
            jsonResponse.referenceRange = firstResult.referenceRange;
            jsonResponse.labTableDivId = labTableDivId;
            jsonResponse.units = firstResult.units;
            jsonResponse.minDateFormatted = Ext.util.Format.dateRenderer('M d, Y H:i')(new Date(jsonResponse.minDate));
            jsonResponse.maxDateFormatted = Ext.util.Format.dateRenderer('M d, Y H:i')(new Date(jsonResponse.maxDate));

            var dataStore = new Ext.data.JsonStore({
                fields: [ {name:'resultDate', type:'date'}, 'result', 'specimen', 'units', 'testName', 'hiLoFlag', 'referenceRange', 'comment'],
                data:jsonResponse.guiLabResults,
                sortInfo: {
                    field: 'resultDate',
                    direction: 'ASC'
                }
            });

            jsonResponse.labTableData = generateData(dataStore.getRange());

            var labTableDivId = "labtable_" + this.window.id;

            var tpl = new Ext.XTemplate
(                   '<div class="labtable">',
                    '<div><b>Title:</b>&nbsp;{labTitle}</div>',
                    '<div><b>Reference Range:</b> {referenceRange}</div>',
                    '<div id="disclaimer_{labTableDivId}" class="disclaimer"></div>',
                    '<table width="100%">',
                        '<thead>',
                            '<tr>',
                                '<th scope="col" class="table-header">Test Date</th>',
                                '<th scope="col" class="table-header">Test Name</th>',
                                '<th scope="col" class="table-header">Specimen</th>',
                                '<th scope="col" class="table-header">Result</th>',
                                '<th scope="col" class="table-header">Units</th>',
                                '<th scope="col" class="table-header">Abnormal</th>',
                                '<th scope="col" class="table-header">Reference Range</th>',
                            '</tr>',
                        '</thead>',
                        '<tbody>',
                            '<tpl for="labTableData">',
                            '<tr bgcolor="{bgColor}">',
                                '<td title="{comment}" style="color:{[values.abnFlag == true ? "white" : "black"]}">{date}</td>',
                                '<td title="{comment}" style="color:{[values.abnFlag == true ? "white" : "black"]}">{testName}</td>',
                                '<td title="{comment}" style="color:{[values.abnFlag == true ? "white" : "black"]}">{specimen}</td>',
                                '<td title="{comment}" style="color:{[values.abnFlag == true ? "white" : "black"]}">{result}</td>',
                                '<td title="{comment}" style="color:{[values.abnFlag == true ? "white" : "black"]}">{units}</td>',
                                '<td title="{comment}" style="color:{[values.abnFlag == true ? "white" : "black"]}">{hiLoFlag}</td>',
                                '<td title="{comment}" style="color:{[values.abnFlag == true ? "white" : "black"]}">{referenceRange}</td>',
                            '</tr>',
                            '</tpl>',
                            '<tr>',
                                '<td class="tablemetadata" colspan="7">',
                                    '<div><b>Units:</b> {units}</div>',
                                    '<div><b>Minimum:</b> {min} - {minDateFormatted}</div>',
                                    '<div><b>Maximum:</b> {max} - {maxDateFormatted}</div>',
                                    '<div><b>Number of Results:</b> {count}</div>',
                                    '<div><b>Average:</b> {avg}</div>',
                                '</td>',
                            '</tr>',
                        '</tbody>',
                    '</table>',
                '</div>'
            );

            this.window.body.dom.innerHTML = tpl.apply(jsonResponse);

            if (jsonResponse.exceptionFlag == true)
            {
                Ext.fly('disclaimer_'+labTableDivId).dom.innerHTML = "<b>Note: Only numeric results have been displayed. Other results are available that have not been numerically resulted.</b>";
            }
        }

        this.window.doLayout();
	}
});
