/*
 * Janus 4.0 (c)
 * Copyright (c) 2011 Hawaii Resource Group LLC. All Rights Reserved.
 * Developed for the Pacific Telehealth & Technology Hui and the Pacific Joint Information Technology Center
 * Contributors:
 *             Honorable Senator Daniel K. Inouye
 *             VA Pacific Islands Health Care System
 *             Tripler Army Medical Center
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with the License.
 *
 * You may obtain a copy of the License at:
 *
 *            http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and limitations under the License.
 */

/**
 * Lab Results Window
 * @param {} config Class configuration object
 *
 * @cfg patientToken
 *     Patient's security token
 * @cfg recordDetailsToken
 * 		Record security token
 */
Ext.hui.desktopwindow.patient.LabResults = Ext.extend(Ext.hui.desktopwindow.DesktopWindow,
{
    initComponent: function() {
		if (!this.title || this.title.length < 0)
        {
            this.title = "Lab Results";
        }
	    this.width = 900;
        this.height = 500;
        this.cls = "labresults";
        this.autoScroll = true;

        this.on('afterrender', function(){this.fetchData()}, this);

        this.addTool({
            id: 'copy',
			qtip: 'Copy to Clipboard',
            cls: 'x-tool-copy',
            overCls:'x-tool-copy-over',
			handler: function (event,toolEl, window){
				if (window) {
                    Ext.hui.common.copyWindow(window);
				}
			}
        });
        this.addTool({
            id: 'print',
			qtip: 'Print',
			handler: function (event,toolEl, window){
				Ext.ux.Printer.BaseRenderer.prototype.stylesheetPath = 'css/print.css';
				if (window) {
					Ext.ux.Printer.printPatInfo(window);
				}
			}
        });
	    Ext.hui.desktopwindow.patient.LabResults.superclass.initComponent.call(this);
	},

	fetchData: function()
	{
		this.getUpdater().setRenderer(new Ext.hui.desktopwindow.patient.LabResultsRenderer(this));
		this.load({
			url: 'rest/patientLabResults.json',
			method:'GET',
			params: {patToken:this.patientToken, recToken:this.recordDetailsToken},
			discardUrl: true,
		    nocache: true,
            disableCaching:true,
		    text: 'Loading...',
		    timeout:Ext.hui.common.CONNECTION_TIMEOUT_SECONDS,
		    scripts: false,
		    scope:this,
	    	callback:function(options, isSuccess, response)
			{
				if (!isSuccess)
				{
					Ext.hui.common.handleNetworkFailure(response.status, this);
				}
			}
		});
    }
});

Ext.reg('labreport_window', Ext.hui.desktopwindow.patient.LabResults);

/**
 * Problem details renderer, parses json response and sends it to the widgets initLayout function.
 */
Ext.hui.desktopwindow.patient.LabResultsRenderer = function(window)
{
	this.window = window;
	Ext.hui.desktopwindow.patient.LabResultsRenderer.superclass.constructor.call(this);
}

Ext.extend(Ext.hui.desktopwindow.patient.LabResultsRenderer, Ext.Updater.BasicRenderer,
{
    render: function(el, xhr, updateManager, callback)
	{
        if (!this.window || this.window.isDestroy) return;

        var jsonResponse = Ext.util.JSON.decode(xhr.responseText);
		el.dom.innerHTML = "";
        var html = "";

        if (jsonResponse.length < 1)
        {
            html = "No lab results found.";
        }
        else
        {

            var data = jsonResponse[0];

            var tplHead = new Ext.XTemplate
            (
                '<table width="100%">',
                    '<thead>',
                        '<tr>',
                          '<td><span class="results-header">Date:</span>&nbsp;{resultDate}</td>',
                          '<td colspan="6"><span class="results-header">Accession Num:</span>&nbsp;{accession}</td>',
                        '</tr>',
                    '</thead>',
                '</table>',
                '<table>',
                    '<tr>',
                        '<td>',
                            '<b>Comment:</b><br/>{comment}<br/>',
                        '</td>',
                    '</tr>',
                '</table>',
                '<table width="100%">',
                    '<thead>',
                        '<tr>',
                            '<th scope="col" class="table-header">Test</th>',
                            '<th scope="col" class="table-header">Histogram</th>',
                            '<th scope="col" class="table-header">Specimen</th>',
                            '<th scope="col" class="table-header">Result</th>',
                            '<th scope="col" class="table-header">Units</th>',
                            '<th scope="col" class="table-header">Abnormal</th>',
                            '<th scope="col" class="table-header">Reference Range</th>',
                        '</tr>',
                    '</thead>',
                    '<tbody>'
            );

            html += tplHead.apply(data);

            for(var i = 0; i < jsonResponse.length; i++)
            {
                data = jsonResponse[i];

                var test = data.printName;

                if (data.testName && data.testName.length > 0)
                {
                    test = data.testName;
                }

                var bgcolor = "white";
                var abnFlag = false;
                if (data.hiLoFlag && data.hiLoFlag.length > 0)
                {
                    bgcolor="#FFCCCC";
                    if (data.hiLoFlag.indexOf("*") > 0)
                    {
                        bgcolor = "red";
                        abnFlag = true;
                    }
                }
                var certifyinit = "";
                var certifydt = "\nCertified: " + data.resultDate;

                if (data.verifiedBy && data.verifiedBy.length > 0)
                {
                    certifyinit =  ":" + data.verifiedBy;
                }

                var certify = certifydt  + certifyinit;

                var resultNum = new Number(data.result);

                data.test = test;
                data.bgcolor = bgcolor;
                data.abnFlag = abnFlag;
                data.resultNum = resultNum;
                data.certify = certify;

                var tplTable = new Ext.XTemplate
                (
                    '<tr bgcolor="{bgcolor}">',
                       '<td style="color:{[values.abnFlag == true ? "white" : "black"]}">',
                            '{test}',
                        '</td>',
                        '<td style="color:{[values.abnFlag == true ? "white" : "black"]}">',
                             '<tpl if="histogramAvailable">',
                                '<span class="histogram-link" id="histogram-graph-link-{testId}">graph</span>',
                                '&nbsp;|&nbsp;<span class="histogram-link" id="histogram-table-link-{testId}">table</span>',
                             '</tpl>',
                             '<tpl if="!histogramAvailable">',
                                'Unavailable',
                             '</tpl>',
                        '</td>',
                        '<td style="color:{[values.abnFlag == true ? "white" : "black"]}">',
                            '{specimen}',
                        '</td>',
                        '<td style="color:{[values.abnFlag == true ? "white" : "black"]}">',
                            '{result}',
                        '</td>',
                        '<td style="color:{[values.abnFlag == true ? "white" : "black"]}">',
                            '{units}',
                        '</td>',
                        '<td style="color:{[values.abnFlag == true ? "white" : "black"]}">',
                            '{hiLoFlag}',
                        '</td>',
                        '<td style="color:{[values.abnFlag == true ? "white" : "black"]}">',
                            '{referenceRange}',
                        '</td>',
                    '</tr>'
                );

                html += tplTable.apply(data);
            }

            html += "</tbody></table>";

            html += "<table><tr><td style='padding:10px;'><span>L=Low | H=High | *=Critical | R=Resist | S=Susc | MS=Mod Susc | I=Intermed | []=Uncert <br>/A=Amended | Comments=  (O)rder, (I)nterpretations, (R)esult</span></td></tr></table>";

        }

        this.window.body.dom.innerHTML = html;
        this.window.doLayout();

        var showLabGraph = function()
        {
            this.desktop.createWindow({
                title:String.format("Lab Graph ~ {0} {1}",
                                                Ext.hui.common.formatSiteMonikerDisplay(this.siteAgency, this.siteMoniker),
                                                this.testName),
                labResult:this
            }, Ext.hui.desktopwindow.patient.LabGraph).show();
        }

        var showLabTable = function()
        {
            this.desktop.createWindow({
                 title:String.format("Lab Table ~ {0} {1}",
                    Ext.hui.common.formatSiteMonikerDisplay(this.siteAgency, this.siteMoniker),
                    this.testName),
                labResult:this,
                desktop:this.desktop
            }, Ext.hui.desktopwindow.patient.LabTable).show();
        }

        for(var i = 0; i < jsonResponse.length; i++)
        {
            var labResultData = jsonResponse[i];

            if (labResultData.histogramAvailable)
            {
                labResultData.patientToken = this.window.patientToken;
                labResultData.desktop = this.window.desktop;
                labResultData.siteMoniker = this.window.siteMoniker;
                Ext.EventManager.addListener('histogram-graph-link-'+labResultData.testId, 'click', showLabGraph, labResultData);
                Ext.EventManager.addListener('histogram-table-link-'+labResultData.testId, 'click', showLabTable, labResultData);
            }
        }
	}
});
