/*
 * Janus 4.0 (c)
 * Copyright (c) 2011 Hawaii Resource Group LLC. All Rights Reserved.
 * Developed for the Pacific Telehealth & Technology Hui and the Pacific Joint Information Technology Center
 * Contributors:
 *             Honorable Senator Daniel K. Inouye
 *             VA Pacific Islands Health Care System
 *             Tripler Army Medical Center
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with the License.
 *
 * You may obtain a copy of the License at:
 *
 *            http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and limitations under the License.
 */

/**
 * Lab Graph Window
 * @param {} config Class configuration object
 *
 */
Ext.hui.desktopwindow.patient.LabGraph = Ext.extend(Ext.hui.desktopwindow.DesktopWindow,
{
    initComponent: function() {
		if (!this.title || this.title.length < 0)
        {
            this.title = "Lab Graph";
        }
	    this.width = 900;
        this.height = 550;
        this.cls = "labgraph";
        this.autoScroll = true;

        this.on('afterrender', function(){this.fetchData()}, this);

        this.addTool({
            id: 'print',
			qtip: 'Print',
			handler: function (event,toolEl, window){
				Ext.ux.Printer.BaseRenderer.prototype.stylesheetPath = 'css/print.css';
				if (window) {
					Ext.ux.Printer.printPatInfo(window);
				}
			}
        });
	    Ext.hui.desktopwindow.patient.LabGraph.superclass.initComponent.call(this);
	},

	fetchData: function()
	{
		this.getUpdater().setRenderer(new Ext.hui.desktopwindow.patient.LabGraphRenderer(this));
		this.load({
			url: 'rest/patientLabResultHistogram.json',
			method:'GET',
			params: {
                patToken:this.labResult.patientToken,
                recToken:this.labResult.histogramRecordToken
            },
			discardUrl: true,
		    nocache: true,
            disableCaching:true,
		    text: 'Loading...',
		    timeout:Ext.hui.common.CONNECTION_TIMEOUT_SECONDS,
		    scripts: false,
		    scope:this,
	    	callback:function(options, isSuccess, response)
			{
				if (!isSuccess)
				{
					Ext.hui.common.handleNetworkFailure(response.status, this);
				}
			}
		});
	}

});

Ext.reg('labgraph_window', Ext.hui.desktopwindow.patient.LabGraph);

/**
 * Lab graph renderer, parses json response and sends it to the widgets initLayout function.
 *
 * @param window The window to render into.
 *
 */
Ext.hui.desktopwindow.patient.LabGraphRenderer = function(window)
{
	this.window = window;
	Ext.hui.desktopwindow.patient.LabGraphRenderer.superclass.constructor.call(this);
}

Ext.extend(Ext.hui.desktopwindow.patient.LabGraphRenderer, Ext.Updater.BasicRenderer,
{
    render: function(el, xhr, updateManager, callback)
	{
        if (!this.window || this.window.isDestroy) return;

        var jsonResponse = Ext.util.JSON.decode(xhr.responseText);
        var generateData = function(records)
        {
            var data = [];

            for(var i = 0; i < records.length; i++)
            {
                var labResult = records[i];
                data.push
                ([
                    //flot graph lib always displays timestamps according to UTC
                    Ext.hui.common.createUTCDate(labResult.data.resultDate).getTime(),
                    parseFloat(labResult.data.result)
                ]);
            }

            return data;
        }

		el.dom.innerHTML = "";
        var labGraphDivId = "labgraph_" + this.window.id;


		if (jsonResponse.guiLabResults == null || jsonResponse.guiLabResults.length < 1)
        {
            this.window.body.dom.innerHTML = '<div class="errortext">An error occurred when attempting to retrieve data for this lab result series.</div>';
        }
        else
        {

            var tpl = new Ext.XTemplate
            (
                 '<div class="labgraphContainer">',
                    '<div><b>Title:</b>&nbsp;{labTitle}</div>',
                    '<div><b>Reference Range:</b> {referenceRange}</div>',
                    '<div class="graph-container" id="{labGraphDivId}"></div>',
                    '<div id="zoomNav-instructions">Use mouse wheel to zoom, click and drag to pan.</div>',
                    '<div id="disclaimer_{labGraphDivId}" class="disclaimer"></div>',
                    '<div id="graphZoomNavContainer" class="zoomNavContainer">',
                        '<div id="zoomNav-{labGraphDivId}"></div>',
                    '</div>',
                    '<div><b>Units:</b> {units}</div>',
                    '<div><b>Minimum:</b> {min} - {minDateFormatted}</div>',
                    '<div><b>Maximum:</b> {max} - {maxDateFormatted}</div>',
                    '<div><b>Number of Results:</b> {count}</div>',
                    '<div><b>Average:</b> {avg}</div>',
                '</div>'
            );
            var firstResult = jsonResponse.guiLabResults[0];
            jsonResponse.labTitle = firstResult.testName;
            jsonResponse.referenceRange = firstResult.referenceRange;
            jsonResponse.labGraphDivId = labGraphDivId;
            jsonResponse.units = firstResult.units;
            jsonResponse.minDateFormatted = Ext.util.Format.dateRenderer('M d, Y H:i')(new Date(jsonResponse.minDate))
            jsonResponse.maxDateFormatted = Ext.util.Format.dateRenderer('M d, Y H:i')(new Date(jsonResponse.maxDate))
            var html = tpl.apply(jsonResponse);

            this.window.body.dom.innerHTML = html;

            if (jsonResponse.exceptionFlag == true)
            {
                Ext.fly('disclaimer_'+labGraphDivId).dom.innerHTML = "<b>Note: Only numeric results have been displayed. Other results are available that have not been numerically resulted.</b>";
            }

            var dataStore = new Ext.data.JsonStore({
                fields: [ {name:'resultDate', type:'date'}, 'result'],
                data:jsonResponse.guiLabResults,
                sortInfo: {
                    field: 'resultDate',
                    direction: 'ASC'
                }
            });


            var dataVals = generateData(dataStore.getRange());

            var plot = $.plot($("#"+labGraphDivId), [ dataVals],
            {
                xaxis:
                {
                    mode: "time",
                    timeformat: "%b %0d, %y"
                },
                series:
                {
                    lines: { show: true },
                    points: { show: true }
                },
                grid:
                {
                    hoverable: true,
                    clickable: true
                },
                zoom:
                {
                    interactive: true
                },
                pan:
                {
                    interactive: true
                }
            });

            // add zoom out button
            $('<div role="button" class="button zoom-buttons", style="top:20px">zoom out</div>').appendTo($("#zoomNav-"+labGraphDivId)).click(function (e) {
                e.preventDefault();
                plot.zoomOut();
            });

            // little helper for taking the repetitive work out of placing
            // panning arrows
            var addArrow = function(dir, right, top, offset) {
                $('<span role="button" class="button pan-arrow-'+dir+'" style="right:' + right + 'px;top:' + top + 'px"></span>').appendTo($("#zoomNav-"+labGraphDivId)).click(function (e) {
                    e.preventDefault();
                    plot.pan(offset);
                });
            }

            addArrow('left', 55, 60, { left: -100 });
            addArrow('right', 25, 60, { left: 100 });
            addArrow('up', 40, 45, { top: -100 });
            addArrow('down', 40, 75, { top: 100 });

            // add zoom out button
            $('<div role="button" class="button zoom-buttons" style="top:100px;">zoom in</div>').appendTo($("#zoomNav-"+labGraphDivId)).click(function (e) {
                e.preventDefault();
                plot.zoom();
            });

            var showGraphTooltip = function(x, y, contents) {
                $('<div class="graph_toolTip" id="'+labGraphDivId+'_tooltip">' + contents + '</div>').css( {
                    position: 'absolute',
                    display: 'none',
                    'text-align':'center',
                    top: y-50,
                    left: x-50,
                    padding: '2px',
                    opacity: 0.80
                }).appendTo("body").fadeIn(200);
            }
            var me = this;
            var previousPoint = null;
            $("#"+labGraphDivId).bind("plothover", function (event, pos, item) {

                if (item) {
                    if (previousPoint != item.dataIndex) {
                        previousPoint = item.dataIndex;

                        $("#"+labGraphDivId+"_tooltip").remove();
                        var x = item.datapoint[0],
                            y = item.datapoint[1];

                        var records = dataStore.getRange();
                        var idx = item.dataIndex;
                        var record = records[idx];

                        //flot graph lib always displays timestamps according to UTC
                        if (record.data.result == y && Ext.hui.common.createUTCDate(record.data.resultDate).getTime() == x)
                        {
                            showGraphTooltip(item.pageX, item.pageY,
                                record.data.result +
                                "<br/>" +
                                Ext.util.Format.dateRenderer('M d, Y H:i')(record.data.resultDate));
                        }
                    }
                }
                else {
                    $("#"+labGraphDivId+"_tooltip").remove();
                    previousPoint = null;
                }
            });

            this.window.doLayout();
        }
	}
});
