/*
 * Janus 4.0 (c)
 * Copyright (c) 2011 Hawaii Resource Group LLC. All Rights Reserved.
 * Developed for the Pacific Telehealth & Technology Hui and the Pacific Joint Information Technology Center
 * Contributors:
 *             Honorable Senator Daniel K. Inouye
 *             VA Pacific Islands Health Care System
 *             Tripler Army Medical Center
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with the License.
 *
 * You may obtain a copy of the License at:
 *
 *            http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and limitations under the License.
 */

/**
 * Patient ClinicalReminders Window
 * @param {} config Class configuration object
 * 
 * @cfg patientToken
 *     Patient's security token
 */
Ext.hui.desktopwindow.patient.ClinicalReminders = Ext.extend(Ext.hui.desktopwindow.DesktopWindow,
{
	initComponent: function() {

        this.sortInfo = {
            field: 'dueDate',
            direction: 'DESC'
        };

        this.dataStore = new Ext.data.Store({
			restful:true,
			proxy: new Ext.data.HttpProxy({
						url: 'rest/patientClinicalReminders.json',
						disableCaching:true,
                        timeout: Ext.hui.common.CONNECTION_TIMEOUT_MS,
						method:'GET'
	    			}),
            baseParams:
            {
                patToken:this.patientToken
            },
	        reader: this.createJsonReader(),
	        
	        sortInfo: this.sortInfo,
			listeners:
			{
				exception:{scope:this, fn:function(dataProxy, type, action, options, response, arg){
												Ext.hui.common.handleNetworkFailure(response.status, this);
                }},
                beforeLoad:
                {
                    scope:this,
                    fn:function()
                    {
                        if (!this || this.isDestroy) return;
                        this.loading = new Ext.LoadMask(this.gridArea.el, {msg:"Loading...", store:this.dataStore});
                        this.loading.show();
                    }
                },
				load:
				{
					scope:this, 
					fn:
					  	function(store, records, options){
		    				  if (!this || this.isDestroy) return;
                              this.setTitle("Clinical Reminders ("+records.length+")");
	    				}
    			},
                datachanged:
                {
                    scope:this,
                    fn:function(store)
                    {
                        if (!this || this.isDestroy) return;
                        this.tableExpandToggleText = Ext.hui.common.EXPAND_TABLE_TEXT;

                        var recs = store.getRange();
                        var jsonDt = [];
                        for(var i = 0; i < recs.length; i++)
                        {
                            jsonDt[i] = recs[i].json;
                        }

                        var inMemoryStore = new Ext.data.Store({
                            proxy: new Ext.ux.data.PagingMemoryProxy(jsonDt),
                            remoteSort:true,
                            reader: this.createJsonReader() ,
                            sortInfo: this.sortInfo
                        });

                        //destroy previous grid panel
                        if (this.gridPanel)
                        {
                            this.gridPanel.destroy();
                        }
                        this.gridPanel = this.createGrid(inMemoryStore, this, Ext.hui.common.WINDOW_NUM_RECORDS_PER_PAGE);
                        inMemoryStore.load({params:{start:0, limit:Ext.hui.common.WINDOW_NUM_RECORDS_PER_PAGE}});

                        this.gridArea.removeAll();
                        this.gridArea.add(this.gridPanel);
                        this.doLayout();
                    }
                }
			},
	        autoLoad: true
	    });


	    
	    this.title='ClinicalReminders';
        this.width = 720;
        this.height = 500;
        this.iconCls = 'icon-grid';
        this.animCollapse = false;
        this.constrainHeader = true;
        this.draggable = true;
        this.layout = 'fit';
        this.dataStore = this.dataStore;
	    this.tools = 
	    [
			{
			id: 'copy',
			qtip: 'Copy to Clipboard',
			handler: function (event,toolEl, panel){
				if (panel.gridPanel) {
					 Ext.hui.common.copyGridPanel(panel.gridPanel);
				}
			}
		},
		{
			id: 'print',
			qtip: 'Print',
			handler: function (event,toolEl, panel){
				Ext.ux.Printer.BaseRenderer.prototype.stylesheetPath = 'css/print.css';
				if (panel.gridPanel) {
					Ext.ux.Printer.printPatInfo(panel.gridPanel);
				}
			}
		},
			{
			    id:'refresh',
			    qtip: 'Refresh',
			    handler: function(event, toolEl, win){
			        win.dataStore.reload();
			    }
			}
		];

        this.gridArea = new Ext.Container({layout:'fit', region:'center', html:''});
	    this.items = [this.gridArea];
	    
	    Ext.hui.desktopwindow.patient.ClinicalReminders.superclass.initComponent.call(this);
	},
    createJsonReader: function()
    {
        return new Ext.data.JsonReader({
	            root: ''
	        },
			[
				{name: 'dueDate', type:'date'},
                {name: 'lastDate', type:'date'},
				{name: 'description'},
				{name: 'clinicalReminderId'},
				{name: 'siteCode'},
                {name: 'recordDetailsToken'},
				{name: 'siteMoniker'},
                {name: 'siteAgency'},
                {name: 'sourcePlatform'}
	        ]);
    },
    createGrid: function(gridDataStore, refWin, recordsPerPage)
    {
        var dateRenderer = Ext.util.Format.dateRenderer('M d, Y');
        var dateTimeRenderer = Ext.util.Format.dateRenderer('M d, Y H:i');

		return new Ext.grid.GridPanel({
	        store: gridDataStore,
	        columns: [
	            {
	                header   : 'Site',
	                width    : 60,
	                sortable : true,
	                dataIndex: 'siteMoniker',
	                renderer :
	                        function(value, metaData, record, rowIndex, colIndex, store) {
	                            return Ext.hui.common.formatSiteMonikerDisplay(record.data.siteAgency, record.data.siteMoniker);
	                        }
	            },
	            {
	                width    : 120,
	                header   : 'Due Date',
	                sortable : true,
	                dataIndex: 'dueDate',
	                renderer: dateRenderer
	            },
                {
	                width    : 150,
	                header   : 'Reminder',
	                sortable : true,
	                dataIndex: 'description',
	                renderer :{
	                        fn:function(value, metaData, record, rowIndex, colIndex, store) {
                                    if (record.data.recordDetailsToken != null)
                                    {
                                        return String.format('<span class="record-details" id="clinicalreminder-details-{0}-{1}">{2}</span>', refWin.id, record.id, record.data.description);
                                    }
                                    else return record.data.description;
                                },
	                        scope:this
	                }
	            },
                {
	                header   : 'Last Recorded',
	                width    : 120,
	                sortable : true,
	                dataIndex: 'lastDate',
                    renderer: dateTimeRenderer
	            }
	        ],
	        listeners:{
	        	rowclick:
	        	{
	        		scope:this,
	        		fn:function(grid, rowIdx, evtObj) {
	        			var record = grid.getStore().getAt(rowIdx);
        				if(evtObj.within(String.format("clinicalreminder-details-{0}-{1}", refWin.id, record.id)))
        				{
        					refWin.desktop.createWindow({
    							title:String.format("Clinical Reminder Details ~ {0} {1}",
                                    Ext.hui.common.formatSiteMonikerDisplay(record.data.siteAgency, record.data.siteMoniker),
                                    record.data.description),
                                patientToken:refWin.patientToken,
    							desktop:refWin.desktop,
    							recordDetailsToken:record.data.recordDetailsToken
    						}, Ext.hui.desktopwindow.patient.ClinicalReminderDetails).show();
        				}
	        		}
	        	},
                viewready:
                {
                    scope:refWin,
                    fn:function()
                    {
                        if (!this || this.isDestroy) return;
                        this.loading.hide();
                    }
                }
	        },
            selModel: new Ext.grid.CellSelectionModel(),
	        stripeRows: true,
            columnLines:true,
	        autoExpandColumn:'description',
	        loadMask:true,
            bbar: refWin.createPagingToolbar(gridDataStore, recordsPerPage),
	        viewConfig: {
	        	emptyText:"No results",
                forceFit:true
	        },
	        cls:"ClinicalReminders"
	    });
    }
});

Ext.reg('clinicalreminders_window', Ext.hui.desktopwindow.patient.ClinicalReminders);