/*
 * Janus 4.0 (c)
 * Copyright (c) 2011 Hawaii Resource Group LLC. All Rights Reserved.
 * Developed for the Pacific Telehealth & Technology Hui and the Pacific Joint Information Technology Center
 * Contributors:
 *             Honorable Senator Daniel K. Inouye
 *             VA Pacific Islands Health Care System
 *             Tripler Army Medical Center
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with the License.
 *
 * You may obtain a copy of the License at:
 *
 *            http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and limitations under the License.
 */

/**
 * Patient Appointments Window
 * @param {} config Class configuration object
 * 
 * @cfg patientToken
 *     Patient's security token
 */
Ext.hui.desktopwindow.patient.Appointments = Ext.extend(Ext.hui.desktopwindow.DesktopWindow,
{
	initComponent: function() {

        this.sortInfo =  {
            field: 'appointmentDate',
            direction: 'ASC'
        };

        this.dataStore = new Ext.data.Store({
			restful:true,
			proxy: new Ext.data.HttpProxy({
						url: 'rest/patientAppointments.json',
						disableCaching:true,
                        timeout: Ext.hui.common.CONNECTION_TIMEOUT_MS,
						method:'GET'
	    			}),
            baseParams:
            {
                patToken:this.patientToken,
                startDate:Ext.util.Format.dateRenderer("m/d/y")(this.startDate),
                endDate:Ext.util.Format.dateRenderer("m/d/y")(this.endDate)
            },
	        reader: this.createJsonReader(),
	        
	        sortInfo: this.sortInfo,
			listeners:
			{
				exception:{scope:this, fn:function(dataProxy, type, action, options, response, arg){
												Ext.hui.common.handleNetworkFailure(response.status, this);
                }},
                beforeLoad:
                {
                    scope:this,
                    fn:function()
                    {
                        if (!this || this.isDestroy) return;
                        this.loading = new Ext.LoadMask(this.gridArea.el, {msg:"Loading...", store:this.dataStore});
                        this.loading.show();
                        if (this.filter) this.filter.disable();
                    }
                },
				load:
				{
					scope:this, 
					fn:
                    function(store, records, options){
		    			if (!this || this.isDestroy) return;
                        if (this.filter)
                        {
                            this.filter.loadWithData();
                            this.filter.enable();
                        }
                       this.setTitle("Appointments ("+records.length+")");
                    }
    			},
                datachanged:
                {
                    scope:this,
                    fn:function(store)
                    {
                        if (!this || this.isDestroy) return;
                        this.tableExpandToggleText = Ext.hui.common.EXPAND_TABLE_TEXT;

                        var recs = store.getRange();
                        var jsonDt = [];
                        for(var i = 0; i < recs.length; i++)
                        {
                            jsonDt[i] = recs[i].json;
                        }

                        var inMemoryStore = new Ext.data.Store({
                            proxy: new Ext.ux.data.PagingMemoryProxy(jsonDt),
                            remoteSort:true,
                            sortInfo: this.sortInfo,
                            reader: this.createJsonReader()
                        });

                        //destroy previous grid panel
                        if (this.gridPanel)
                        {
                            this.gridPanel.destroy();
                        }
                        this.gridPanel = this.createGrid(inMemoryStore, this, Ext.hui.common.WINDOW_NUM_RECORDS_PER_PAGE);
                        inMemoryStore.load({params:{start:0, limit:Ext.hui.common.WINDOW_NUM_RECORDS_PER_PAGE}});

                        this.gridArea.removeAll();
                        this.gridArea.add(this.gridPanel);
                        this.doLayout();
                        if (this.filter) this.filter.displayFilter(true);
                    }
                }
			},
	        autoLoad: true
	    });
		
		this.filter = new Ext.hui.filter.AppointmentsFilter({
            region:'north',
            dataStore:this.dataStore,
            startDate:this.startDate,
            endDate:this.endDate,
            disabled:true,
            hidden:true
        });


	    
	    this.title='Appointments';
        this.width = 900;
        this.height = 450;
        this.layout = "border";
        this.iconCls = 'icon-grid';
        this.animCollapse = false;
        this.constrainHeader = true;
        this.draggable = true;
        this.dataStore = this.dataStore;
	    this.tools = 
	    [{
			id: 'filter',
			qtip: 'Configure Filter',
            cls: 'x-tool-filter',
            overCls:'x-tool-filter-over',
			handler: function (event,toolEl, panel){
				if (panel.filter) {
                    panel.filter.toggleView();
				}
			}
		},
        {
			id: 'copy',
			qtip: 'Copy to Clipboard',
			handler: function (event,toolEl, panel){
				if (panel.gridPanel) {
					 Ext.hui.common.copyGridPanel(panel.gridPanel);
				}
			}
		},
		{
			id: 'print',
			qtip: 'Print',
			handler: function (event,toolEl, panel){
				Ext.ux.Printer.BaseRenderer.prototype.stylesheetPath = 'css/print.css';
				if (panel.gridPanel) {
					Ext.ux.Printer.printPatInfo(panel.gridPanel);
				}
			}
		},
			{
			    id:'refresh',
			    qtip: 'Refresh',
			    handler: function(event, toolEl, win){
			        win.dataStore.reload();
			    }
			}
		];
        this.gridArea = new Ext.Container({layout:'fit', region:'center', html:''});
	    this.items = [this.filter, this.gridArea];
	    
	    Ext.hui.desktopwindow.patient.Appointments.superclass.initComponent.call(this);
	},
    createJsonReader: function()
    {
        return new Ext.data.JsonReader({
	            root: ''
	        },
			[
				{name: 'appointmentDate', type:'date'},
                {name: 'clinic'},
                {name: 'clinicId'},
				{name: 'status'},
                {name: 'type'},
				{name: 'siteCode'},
                {name: 'providerName'},
                {name: 'reason'},
                {name: 'appointmentId'},
                {name: 'recordDetailsToken'},
				{name: 'siteMoniker'},
                {name: 'siteAgency'},
                {name: 'sourcePlatform'}
	        ]);
    },
    createGrid: function(gridDataStore, refWin, recordsPerPage)
    {
        var dateRenderer = Ext.util.Format.dateRenderer('M d, Y H:i');

		return new Ext.grid.GridPanel({
	        store: gridDataStore,
	        columns: [
	        	{
	                header   : 'Site',
	                width    : 120,
	                sortable : true,
	                dataIndex: 'siteMoniker',
	                renderer :
	                        function(value, metaData, record, rowIndex, colIndex, store) {
	                            return Ext.hui.common.formatSiteMonikerDisplay(record.data.siteAgency, record.data.siteMoniker);
	                        }
	            },
	            {
                    id: 'appointmentDate',
	                width    : 120,
	                header   : 'Date',
	                sortable : true,
	                dataIndex: 'appointmentDate',
	                renderer:{
	                        fn:function(value, metaData, record, rowIndex, colIndex, store) {
	                            if (record.data.recordDetailsToken != null)
	                            {
	                            	return String.format('<span class="record-details" id="appointment-details-{0}-{1}">{2}</span>', refWin.id, record.id, dateRenderer(record.data.appointmentDate));
	                            }
	                            else return dateRenderer(record.data.appointmentDate);

	                        },
	                        scope:this
                    }
	            },
                {
	                width    : 120,
	                header   : 'Clinic',
	                sortable : true,
	                dataIndex: 'clinic'
	            },
                {   width    : 100,
	                header   : 'Status',
	                sortable : true,
	                dataIndex: 'status'
	            },
	        	{
	                id: 'type',
	                width    : 125,
	                header   : 'Type',
	                sortable : true,
	                dataIndex: 'type'

	            },
                {
	                header   : 'Provider',
	                width    : 100,
	                sortable : true,
	                dataIndex: 'providerName'
	            },
                {
	                header   : 'Reason',
	                width    : 100,
	                sortable : true,
	                dataIndex: 'reason'
	            },
                {
	                width    : 20,
	                header   : '',
	                renderer:{
                        fn:function(value, metaData, record, rowIndex, colIndex, store) {
                            if (Ext.hui.app.WRITEBACK_ENABLE=="true"){
                                if (record.data.appointmentId != null && record.data.siteCode == Ext.hui.app.WRITEBACK_SITECODE)
                                {
                                    return String.format('<div class="addNote-button" id="addNote-{0}-{1}">.</div>', refWin.id, record.id);
                                }
                                //else return dateRenderer(record.data.appointmentDate);
                            }
                        },
                        scope:this
                    }
	            }

	        ],
	        listeners:{
	        	rowclick:
	        	{
	        		scope:this,
	        		fn:function(grid, rowIdx, evtObj) {
	        			var record = grid.getStore().getAt(rowIdx);
                        if (Ext.get(String.format("appointment-details-{0}-{1}", refWin.id, record.id))){
                            if(evtObj.within(String.format("appointment-details-{0}-{1}", refWin.id, record.id)))
                            {
                                refWin.desktop.createWindow({
                                    title:String.format("Appointment Details ~ {0} {1}",
                                        Ext.hui.common.formatSiteMonikerDisplay(record.data.siteAgency, record.data.siteMoniker),
                                        record.data.clinic),
                                    patientToken:refWin.patientToken,
                                    desktop:refWin.desktop,
                                    recordDetailsToken:record.data.recordDetailsToken
                                }, Ext.hui.desktopwindow.patient.AppointmentDetails).show();
                            }
                        }
                        if (Ext.get(String.format("addNote-{0}-{1}", refWin.id, record.id))){
                            if(evtObj.within(String.format("addNote-{0}-{1}", refWin.id, record.id)))
                            {
                                refWin.desktop.createWindow({
                                    title:String.format("New Note ~ {0} {1}",
                                        Ext.hui.common.formatSiteMonikerDisplay(record.data.siteAgency, record.data.siteMoniker),
                                        record.data.clinic),
                                    patientToken:refWin.patientToken,
                                    desktop:refWin.desktop,
                                    clinicId: record.data.clinicId
                                }, Ext.hui.desktopwindow.patient.AddNote).show();
                            }
                        }
                    }
	        	},
                viewready:
                {
                    scope:refWin,
                    fn:function()
                    {
                        if (!this || this.isDestroy) return;
                        if (this.filter) this.filter.enable();
                        this.loading.hide();
                    }
                }
	        },
            selModel: new Ext.grid.CellSelectionModel(),
	        stripeRows: true,
            autoExpandColumn:'clinic',
            border:true,
            bodyBorder:true,
            loadMask:true,
            layout:'fit',
            region: 'center',
            columnLines:true,
            bbar: refWin.createPagingToolbar(gridDataStore, recordsPerPage),
            viewConfig: {
                emptyText:"No results",
                forceFit:true
            }
	    });
    }
});

Ext.reg('appointments_window', Ext.hui.desktopwindow.patient.Appointments);