/*
 * Janus 4.0 (c)
 * Copyright (c) 2011 Hawaii Resource Group LLC. All Rights Reserved.
 * Developed for the Pacific Telehealth & Technology Hui and the Pacific Joint Information Technology Center
 * Contributors:
 *             Honorable Senator Daniel K. Inouye
 *             VA Pacific Islands Health Care System
 *             Tripler Army Medical Center
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with the License.
 *
 * You may obtain a copy of the License at:
 *
 *            http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and limitations under the License.
 */

package gov.va.med.dao;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import gov.va.med.guibeans.*;
import gov.va.med.vhahon.common.Utils;


import gov.va.med.jmeadows.webservice.Vitals;

public class VitalsDao extends DaoBase
{
    private static final int SNOMED_PERCENT = 118582008;
    private static final int SNOMED_PAIN_SCALE = 419410004;
    private static final int SNOMED_PER_MIN =  258983007;
    private static final int SNOMED_CELSIUS = 258710007;
    private static final int SNOMED_FAHRENHEIT = 258712004;
    private static final int SNOMED_GRAMS = 258682000;
    private static final int SNOMED_INCHES = 258677007;
    private static final int SNOMED_CENTIMETERS = 258672001;
    private static final int SNOMED_MMHG = 259018001;
    private static final int SNOMED_OUNCES = 258692008;
    private static final int SNOMED_POUNDS = 258693003;
    private static final int SNOMED_KILOGRAMS = 258683005;
    private static final int SNOMED_GRAM = 258682000;

    private static String VISTA_LABEL_VITAL_SIGN = "VITAL SIGN";
    private static String VISTA_LABEL_TEMPERATURE = "TEMPERATURE";
    private static String VISTA_LABEL_PULSE = "PULSE";
    private static String VISTA_LABEL_RESPIRATION = "RESPIRATION";
    private static String VISTA_LABEL_BLOOD_PRESSURE = "BLOOD PRESSURE";
    private static String VISTA_LABEL_HEIGHT = "HEIGHT";
    private static String VISTA_LABEL_WEIGHT = "WEIGHT";
    private static String VISTA_LABEL_PAIN = "PAIN";
    private static String VISTA_LABEL_PULSE_OXIMETRY = "PULSE OXIMETRY";
    private static String VISTA_LABEL_CIRCUMFERENCE_GIRTH = "CIRCUMFERENCE / GIRTH";
    private static String VISTA_LABEL_CENTRAL_VENOUS_PRESSURE = "CENTRAL VENOUS PRESSURE";
    private static String VISTA_LABEL_BODY_MASS_INDEX = "BODY MASS INDEX";

    public List<GUIVitals> getPatientVitals(GUIProvider provider, GUIPatient patient, Calendar startDate, Calendar endDate)
	{
		try {
			Vitals[] vitals = getServiceHandler().getPatientVitals(createQueryBean(provider, patient, startDate, endDate));
			
			List<GUIVitals> guiVitals = new ArrayList<GUIVitals>();
			
			for(Vitals vital : vitals)
			{
				String dateTaken = "";
				String type = "";
				String result = "";
                String ien = "";
                String rateUnitsCode = "";
                String rateUnits = "";
				
				if (vital.getDateTimeTaken() != null)
				{
					dateTaken = Utils.formatDisplayDate(vital.getDateTimeTaken());
				}
				
				if (vital.getVitalType() != null)
				{
					type = vital.getVitalType();
				}
				
				if (vital.getRate() != null)
				{
					result = vital.getRate();
				}

                if (vital.getUnits() != null)
                {
                    rateUnits = vital.getUnits();
                }

                if (vital.getUnitsCode() != null)
                {
                    rateUnitsCode = vital.getUnitsCode();
                }
				
				GUIVitals guiVital = new GUIVitals();
				guiVital.setDateTaken(dateTaken);
				guiVital.setType(type);
				guiVital.setResult(result);
                guiVital.setUnits(rateUnits);
                guiVital.setUnitsCode(rateUnitsCode);

                addSiteDataToGUIBean(guiVital, vital);
				
				guiVitals.add(guiVital);
			}
			
			return guiVitals;
			
		} catch (Exception e) {
			throw new DaoException(e);
		}
	}

   public GUIVitalsGraphResults getVitalsGraphData(GUIProvider provider, GUIPatient patient, String vitalType, String rateUnitsCode, Calendar startDate, Calendar endDate)
    {
        try {
            Vitals[] vitals = getServiceHandler().getPatientVitals(
                    createQueryBean(provider, patient, startDate, endDate));

            return toGUIVitalsGraph(vitals, vitalType, rateUnitsCode);

        } catch (Exception e) {
            throw new DaoException(e);
        }
    }

    /**
     * Get default vital unit code for a certain vital types, otherwise returns an empty string
     * if vital type is not among list
     * @param vitalType
     * @return
     */
    private String getVitalUnitCodeFromVitalType(String vitalType)
    {
        if (vitalType.equalsIgnoreCase(VISTA_LABEL_BLOOD_PRESSURE))
        {
            return "" + SNOMED_MMHG;
        }
        else if (vitalType.equalsIgnoreCase(VISTA_LABEL_TEMPERATURE))
        {
            return "" + SNOMED_FAHRENHEIT;
        }
        else if (vitalType.equalsIgnoreCase(VISTA_LABEL_WEIGHT))
        {
            return "" + SNOMED_POUNDS;
        }
        else if (vitalType.equalsIgnoreCase(VISTA_LABEL_HEIGHT))
        {
            return "" + SNOMED_INCHES;
        }
        else if (vitalType.equalsIgnoreCase(VISTA_LABEL_PULSE))
        {
            return "" + SNOMED_PER_MIN;
        }
        else if (vitalType.equalsIgnoreCase(VISTA_LABEL_RESPIRATION))
        {
            return "" + SNOMED_PER_MIN;
        }
        else if (vitalType.equalsIgnoreCase(VISTA_LABEL_PAIN))
        {
            return "" + SNOMED_PAIN_SCALE;
        }

        return "";
    }

    /**
     * Get vital graphing results.
     *
     * Vitals are filtered bu vital type and unit test code. If a unit test code is missing from the
     * vital bean, an attempt is made to find a default unit test code value.
     * If a default value cannot be found, the vital will be included in the filtered list.
     * @param vitals
     * @param vitalType
     * @param rateUnitsCode
     * @return
     */
    private GUIVitalsGraphResults toGUIVitalsGraph(Vitals[] vitals, String vitalType, String rateUnitsCode)
    {
        //determine vital unit code default value if one isn't provided
        if (rateUnitsCode == null || rateUnitsCode.length() < 1)
        {
            rateUnitsCode = getVitalUnitCodeFromVitalType(vitalType);
        }

        //filter out vitals the that are of vital type
        List<Vitals> filteredVitals = new ArrayList<Vitals>();

        for(Vitals vital : vitals)
        {
            //if rateUnitsCode is null or empty, filter only on vital type
            if (vital.getVitalType().equalsIgnoreCase(vitalType) &&
                    (rateUnitsCode == null ||
                     rateUnitsCode.length() < 1))
            {
                filteredVitals.add(vital);
            }
            else if (vital.getVitalType().equalsIgnoreCase(vitalType))
            {
                //grab vital's test unit code.
                String vitalRateUnitsCode = vital.getUnitsCode();
                //set vital test unit code to default if empty
                if (vitalRateUnitsCode == null || vitalRateUnitsCode.length() < 1)
                {
                    vitalRateUnitsCode = getVitalUnitCodeFromVitalType(vital.getVitalType());
                }
                //only include matching test unit codes
                if (rateUnitsCode.equals(vitalRateUnitsCode))
                {
                    filteredVitals.add(vital);
                }
            }
        }

        GUIVitalsGraphResults guiVitalsGraphResults = new GUIVitalsGraphResults();

        int count = 0;

        String rateDate;
        double rateDbl = 0;
        String rateDateStr;
        String rateStr = "";
        double totalRateDbl = 0;
        double minRateDbl = 0;
        double maxRateDbl = 0;
        double avgRateDbl = 0;
        String minDateRate = null;
        String maxDateRate = null;

        String systolicDate;
        double systolicDbl = 0;
        String systolicDateStr;
        String systolicStr = "";
        double totalSystolicDbl = 0;
        double minSystolicDbl = 0;
        double maxSystolicDbl = 0;
        double avgSystolicDbl = 0;
        String minDateSystolic = null;
        String maxDateSystolic = null;

        String diastolicDate;
        double diastolicDbl = 0;
        String diastolicDateStr;
        String diastolicStr = "";
        double totalDiastolicDbl = 0;
        double minDiastolicDbl = 0;
        double maxDiastolicDbl = 0;
        double avgDiastolicDbl = 0;
        String minDateDiastolic = null;
        String maxDateDiastolic = null;

        List<GUIVitalsGraphEntry> guiVitalsGraphEntries = new ArrayList<GUIVitalsGraphEntry>();

        boolean isBloodPressureVital = vitalType.equalsIgnoreCase("BLOOD PRESSURE");

        for(Vitals vital : filteredVitals)
        {
            GUIVitalsGraphEntry guiVitalsGraphEntry = toGUIVitalGraphEntry(vital);

            if (!isBloodPressureVital)
            {
                try
                {
                    rateDateStr = guiVitalsGraphEntry.getDateTimeTaken();
                    rateStr = guiVitalsGraphEntry.getRate();

                    if (rateDateStr == null || rateStr == null) continue;

                     rateDbl = new Double(rateStr).doubleValue();
                     guiVitalsGraphEntries.add(guiVitalsGraphEntry);
                }
                catch (Exception e)
                {
                    //do not include non-numeric results in lab graph result set
                    guiVitalsGraphResults.setExceptionFlag(true);
                    Utils.logMessage("Bad value in vitals result cell: '" + rateStr + "'");
                    continue;
                }
                rateDate = rateDateStr;
                count = count + 1;
                totalRateDbl = totalRateDbl + rateDbl;
                if (count == 1) {
                    minRateDbl = rateDbl;
                    maxRateDbl = rateDbl;
                    minDateRate = rateDate;
                    maxDateRate = rateDate;
                }
                else {
                    if (minRateDbl > rateDbl) {
                        minRateDbl = rateDbl;
                        minDateRate = rateDate;
                    }
                    if (maxRateDbl < rateDbl) {
                        maxRateDbl = rateDbl;
                        maxDateRate = rateDate;
                    }
                }
            }
            else
            {
                try
                {
                    systolicDateStr = guiVitalsGraphEntry.getDateTimeTaken();
                    diastolicDateStr = guiVitalsGraphEntry.getDateTimeTaken();

                    systolicStr = guiVitalsGraphEntry.getSystolicRate();
                    diastolicStr = guiVitalsGraphEntry.getDiastolicRate();

                    if (systolicDateStr == null || diastolicStr == null || diastolicStr == null || diastolicDateStr == null) continue;

                     systolicDbl = new Double(systolicStr).doubleValue();
                     diastolicDbl = new Double(diastolicStr).doubleValue();
                     guiVitalsGraphEntries.add(guiVitalsGraphEntry);
                }
                catch (Exception e)
                {
                    //do not include non-numeric results in lab graph result set
                    guiVitalsGraphResults.setExceptionFlag(true);
                    Utils.logMessage("Bad value in vitals result cell: '" + systolicDbl+"/" + diastolicDbl + "'");
                    continue;
                }

                systolicDate = systolicDateStr;
                diastolicDate = diastolicDateStr;

                count++;
                totalSystolicDbl = totalSystolicDbl + systolicDbl;
                totalDiastolicDbl = totalDiastolicDbl + diastolicDbl;

                if (count == 1) {

                    minSystolicDbl = systolicDbl;
                    maxSystolicDbl = systolicDbl;
                    minDateSystolic = systolicDate;
                    maxDateSystolic = systolicDate;

                    minDiastolicDbl = diastolicDbl;
                    maxDiastolicDbl = diastolicDbl;
                    minDateDiastolic = diastolicDate;
                    maxDateDiastolic = diastolicDate;
                }
                else {

                    if (minSystolicDbl > systolicDbl) {
                        minSystolicDbl = systolicDbl;
                        minDateSystolic = systolicDate;
                    }
                    if (maxSystolicDbl < systolicDbl) {
                        maxSystolicDbl = systolicDbl;
                        maxDateSystolic = systolicDate;
                    }
                    
                    if (minDiastolicDbl > diastolicDbl) {
                        minDiastolicDbl = diastolicDbl;
                        minDateDiastolic = diastolicDate;
                    }
                    if (maxDiastolicDbl < diastolicDbl) {
                        maxDiastolicDbl = diastolicDbl;
                        maxDateDiastolic = diastolicDate;
                    }
                }
            }
        }
        if (!isBloodPressureVital)
        {
		    if (count > 0) avgRateDbl = totalRateDbl/count;
        }
        else
        {
            if (count > 0)
            {
                avgSystolicDbl = totalSystolicDbl/count;
                avgDiastolicDbl = totalDiastolicDbl/count;
            }
        }

        guiVitalsGraphResults.setVitalEntries(guiVitalsGraphEntries);
        guiVitalsGraphResults.setAvgRate(roundAverage(avgRateDbl));
        guiVitalsGraphResults.setCount(count);
        guiVitalsGraphResults.setMaxRate(maxRateDbl);
        guiVitalsGraphResults.setMinRate(minRateDbl);
        guiVitalsGraphResults.setMinRateDate(minDateRate);
        guiVitalsGraphResults.setMaxRateDate(maxDateRate);
        guiVitalsGraphResults.setTotalRate(totalRateDbl);
        
        guiVitalsGraphResults.setBloodPressureVital(isBloodPressureVital);
        
        guiVitalsGraphResults.setAvgSystolic(roundAverage(avgSystolicDbl));
        guiVitalsGraphResults.setMaxSystolic(maxSystolicDbl);
        guiVitalsGraphResults.setMinSystolic(minSystolicDbl);
        guiVitalsGraphResults.setMinSystolicDate(minDateSystolic);
        guiVitalsGraphResults.setMaxSystolicDate(maxDateSystolic);
        guiVitalsGraphResults.setTotalSystolic(totalSystolicDbl);
        
        guiVitalsGraphResults.setAvgDiastolic(roundAverage(avgDiastolicDbl));
        guiVitalsGraphResults.setMaxDiastolic(maxDiastolicDbl);
        guiVitalsGraphResults.setMinDiastolic(minDiastolicDbl);
        guiVitalsGraphResults.setMinDiastolicDate(minDateDiastolic);
        guiVitalsGraphResults.setMaxDiastolicDate(maxDateDiastolic);
        guiVitalsGraphResults.setTotalDiastolic(totalDiastolicDbl);

        return guiVitalsGraphResults;

    }

    private GUIVitalsGraphEntry toGUIVitalGraphEntry(Vitals vital)
    {
        GUIVitalsGraphEntry guiVitalsGraphEntry = new GUIVitalsGraphEntry();

        guiVitalsGraphEntry.setDateTimeTaken(Utils.formatDisplayDate(vital.getDateTimeTaken()));
        guiVitalsGraphEntry.setType(vital.getVitalType());

        String units = "";
        if (vital.getUnits() != null)
        {
            units = vital.getUnits();
        }

        guiVitalsGraphEntry.setUnits(units);

        addSiteDataToGUIBean(guiVitalsGraphEntry, vital);

        if (vital.getVitalType().equalsIgnoreCase("BLOOD PRESSURE"))
        {
            String bpRate = vital.getRate();
            int bpDividerIdx = bpRate.indexOf("/");
            String systolic = bpRate.substring(0, bpDividerIdx);
            String diastolic = bpRate.substring(bpDividerIdx+1, bpRate.length());

            guiVitalsGraphEntry.setBloodPressureVital(true);
            guiVitalsGraphEntry.setSystolicRate(systolic);
            guiVitalsGraphEntry.setDiastolicRate(diastolic);
        }
        else
        {
            guiVitalsGraphEntry.setRate(vital.getRate());
        }

        return guiVitalsGraphEntry;
    }

    private double roundAverage(Double avg) {
        String s = new Double(avg).toString();
        int i = s.indexOf('.');
        if (i >= 0) {
            int o = 0;
            if (s.length() > i+3) {
                o = new Integer(s.substring(i+3,i+4)).intValue();
                s = s.substring(0,i+3);
            }
            avg = new Double(s).doubleValue();
            if (o > 4) avg += .01;
        }
        return avg;
    }
}
