/*
 * Janus 4.0 (c)
 * Copyright (c) 2011 Hawaii Resource Group LLC. All Rights Reserved.
 * Developed for the Pacific Telehealth & Technology Hui and the Pacific Joint Information Technology Center
 * Contributors:
 *             Honorable Senator Daniel K. Inouye
 *             VA Pacific Islands Health Care System
 *             Tripler Army Medical Center
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with the License.
 *
 * You may obtain a copy of the License at:
 *
 *            http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and limitations under the License.
 */

package gov.va.med.dao;

import gov.va.med.guibeans.GUIPatient;
import gov.va.med.guibeans.GUIProblem;
import gov.va.med.guibeans.GUIProblemNote;
import gov.va.med.guibeans.GUIProvider;
import gov.va.med.vhahon.common.Utils;

import gov.va.med.jmeadows.webservice.Problem;
import gov.va.med.jmeadows.webservice.ProblemDetail;
import gov.va.med.jmeadows.webservice.ProblemNote;

import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

/**
 *
 *
 */
public class ProblemDao extends DaoBase {

    public List<GUIProblem> getPatientProblemList(GUIProvider provider, GUIPatient patient, String status) {
        try {
            Problem[] problems = getServiceHandler().getPatientProblemList(createQueryBean(provider, patient, null, null, status));
            List<GUIProblem> guiProblems = new ArrayList<GUIProblem>();
            for (int i = 0; i < problems.length; i++) {
                Problem problem = problems[i];

                GUIProblem guiProblem = new GUIProblem();

                mapProblemData(problem, guiProblem);

                if (problem instanceof ProblemDetail) {
                    //problems that already contain detail 
                    //information will be cached within the user's session
                    guiProblem.setSessionCachedRecord(true);

                    //set identifier to position iteration position 
                    guiProblem.setProblemId("cached:^" + i);

                    mapProblemDetailsData((ProblemDetail) problem, guiProblem);
                }

                guiProblems.add(guiProblem);
            }

            return guiProblems;

        } catch (Exception ex) {
            throw new DaoException(ex);
        }
    }

    public GUIProblem getProblemDetails(GUIProvider provider, GUIPatient patient, String recordId, String recordSiteCode) {
        try {
            ProblemDetail problemDetail = getServiceHandler().getProblemDetail(createQueryBean(provider, patient, recordId, recordSiteCode));

            GUIProblem guiProblem = null;
            
            if (problemDetail != null) {
                //One GUI problem bean is used to house both general and details information
                //in this case, just populate the details related data of the GUI problem bean
                guiProblem = new GUIProblem();

                mapProblemData(problemDetail, guiProblem);
                mapProblemDetailsData(problemDetail, guiProblem);
            }

            return guiProblem;

        } catch (Exception ex) {
            throw new DaoException(ex);
        }
    }

    private void mapProblemData(Problem problem, GUIProblem guiProblem) {
        String onsetDate = "";
        String icdCode = "";
        String status = "";
        String lastUpdatedDate = "";
        String description = "";
        String priority = "";
        String problemId = "";

        if (problem.getOnsetDate() != null) {
            onsetDate = Utils.formatDisplayDate(problem.getOnsetDate());
        }
        if (problem.getLastModifiedDate() != null) {
            lastUpdatedDate = Utils.formatDisplayDate(problem.getLastModifiedDate());
        }

        if (problem.getIcdCode() != null) {
            icdCode = problem.getIcdCode();
        }

        if (problem.getStatus() != null) {
            
            if (problem.getStatus().equals("A"))
            {
            	status = "Active";
            }
            else if (problem.getStatus().equals("I"))
            {
            	status = "Inactive";
            }
            else
            {
            	status = problem.getStatus();
            }
        }

        if (problem.getDescription() != null) {
            description = problem.getDescription();
        }

        if (problem.getAcuity() != null) {
            priority = problem.getAcuity();
        }

        if (problem.getId() != null) {
            problemId = problem.getId();
        }

        guiProblem.setOnsetDate(onsetDate);
        guiProblem.setIcdCode(icdCode);
        guiProblem.setStatus(status);
        guiProblem.setLastUpdated(lastUpdatedDate);
        guiProblem.setDescription(description);
        guiProblem.setPriority(priority);
        guiProblem.setProblemId(problemId);

        addSiteDataToGUIBean(guiProblem, problem);
    }

    private void mapProblemDetailsData(ProblemDetail problemDetail, GUIProblem guiProblemDetail) {
        String hospitalLocation = "";
        String responsibleProvider = "";
        String enteredDate = "";
        String enteredBy = "";
        String recordedDate = "";
        String recordedBy = "";
        List<GUIProblemNote> guiNotes = new ArrayList<GUIProblemNote>();

        if (problemDetail.getHospitalLocation() != null) {
            hospitalLocation = problemDetail.getHospitalLocation();
        }

        if (problemDetail.getResponsibleProvider() != null) {
            responsibleProvider = problemDetail.getResponsibleProvider();
        }

        if (problemDetail.getEnteredDate() != null) {
            enteredDate = Utils.formatDisplayDate(problemDetail.getEnteredDate());
        }

        if (problemDetail.getEnteredBy() != null) {
            enteredBy = problemDetail.getEnteredBy();
        }

        if (problemDetail.getRecordedDate() != null) {
            recordedDate = Utils.formatDisplayDate(problemDetail.getRecordedDate());
        }

        if (problemDetail.getRecordedBy() != null) {
            recordedBy = problemDetail.getRecordedBy();
        }

        if (problemDetail.getNotes() != null) {
            for (ProblemNote note : problemDetail.getNotes()) {
                String noteDate = "";
                String noteEnteredBy = "";
                String noteText = "";

                if (note.getNoteDate() != null) {
                    noteDate = Utils.formatDisplayDate(note.getNoteDate());
                }

                if (note.getNoteEnteredBy() != null) {
                    noteEnteredBy = note.getNoteEnteredBy();
                }

                if (note.getNoteText() != null) {
                    noteText = note.getNoteText();
                }

                GUIProblemNote guiNote = new GUIProblemNote();
                guiNote.setNoteDate(noteDate);
                guiNote.setNoteEnteredBy(noteEnteredBy);
                guiNote.setNoteText(noteText);

                guiNotes.add(guiNote);
            }
        }

        guiProblemDetail.setHospitalLocation(hospitalLocation);
        guiProblemDetail.setEnteredDate(enteredDate);
        guiProblemDetail.setEnteredBy(enteredBy);
        guiProblemDetail.setRecordedDate(recordedDate);
        guiProblemDetail.setRecordedBy(recordedBy);
        guiProblemDetail.setResponsibleProvider(responsibleProvider);
        guiProblemDetail.setNotes(guiNotes);
    }
}
