/*
 * Janus 4.0 (c)
 * Copyright (c) 2011 Hawaii Resource Group LLC. All Rights Reserved.
 * Developed for the Pacific Telehealth & Technology Hui and the Pacific Joint Information Technology Center
 * Contributors:
 *             Honorable Senator Daniel K. Inouye
 *             VA Pacific Islands Health Care System
 *             Tripler Army Medical Center
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with the License.
 *
 * You may obtain a copy of the License at:
 *
 *            http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and limitations under the License.
 */

package gov.va.med.dao;

import gov.va.med.guibeans.*;
import gov.va.med.jmeadows.webservice.Patient;
import gov.va.med.vhahon.common.Utils;

import gov.va.med.jmeadows.webservice.PatientDemographics;
import gov.va.med.jmeadows.webservice.PatientDemographicsDetail;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import javax.xml.soap.SOAPException;

/**
 *
 *
 */
public class PatientDao extends DaoBase
{
    public List<GUILookupPatient> lookupPatient(GUIProvider provider, String searchQuery, String lookupSiteCode, byte[] encryptSecurityKey)
    {
 
        try {
            Patient[] patients = getServiceHandler().lookupPatient(createQueryBean(provider, searchQuery, lookupSiteCode));
            List<GUILookupPatient> guiLookupPatients = new ArrayList<GUILookupPatient>();
            
            for (Patient patient : patients) {
                String fmpssn = patient.getSSN();
                fmpssn = Utils.formattedSSN(fmpssn);
                String age = patient.getAge();
                if (age == null || age.length() < 1) {
                    age = "N/A";
                }
                String gender = patient.getGender();
                if (gender == null || gender.length() < 1) {
                    gender = "N/A";
                }

                String patientToken = generatePatientToken(encryptSecurityKey, patient.getPatientId(), patient.getPatientIdSiteCode());

                GUILookupPatient guiLookupPatient = new GUILookupPatient();
                guiLookupPatient.setName(patient.getName());
                guiLookupPatient.setAge(age);
                guiLookupPatient.setFmpssn(fmpssn);
                guiLookupPatient.setGender(gender);
                guiLookupPatient.setSensitive(patient.isSensitive());
                guiLookupPatient.setPatientToken(patientToken);

                guiLookupPatients.add(guiLookupPatient);
            }

            return guiLookupPatients;
            
        } catch (Exception ex) {
            throw new DaoException(ex);
        } 
    }

    public GUIPatient selectPatient(GUIProvider provider, String patientId, String patientSiteCode)
    {
        try {
            Patient patient = getServiceHandler().selectPatient(createQueryBean(provider, patientId, patientSiteCode));

            return toGUIPatient(patient);
        } catch (SOAPException ex) {
            throw new DaoException(ex);
        }
    }

    public List<GUIPatientDemographics> getPatientDemographics(GUIProvider provider, GUIPatient patient)
    {
        try {
            List<PatientDemographics> patDemos = getServiceHandler().getPatientDemographics(createQueryBean(provider, patient));
            
            List<GUIPatientDemographics> guiPatDemos = new ArrayList<GUIPatientDemographics>();

            for(PatientDemographics patDemo : patDemos)
            {
            	String name = "";
                String address1= "";
                String address2= "";
                String city= "";
                String state= "";
                String zipCode = "";
                String fmpssn = "";
                String pcm = "";
                String pcmPhone = "";
                String age = "";
                String dob = "";
                String gender = "";
                String phone1 = "";
                String phone2 = "";
                boolean isCodeGray = false;
                String codeGrayMsg = "";
                String percentServiceConnected = "";
                String clinic = "";
                
                if (patDemo.getName() != null)
                {
                	name = patDemo.getName();
                }
                
                if (patDemo.getAddress1() != null)
                {
                	address1 = patDemo.getAddress1();
                }
                
                if (patDemo.getAddress2() != null)
                {
                	address2 = patDemo.getAddress2();
                }
                
                if (patDemo.getCity() != null)
                {
                	city = patDemo.getCity();
                }
                
                if (patDemo.getState() != null)
                {
                	state = patDemo.getState();
                }
                
                if (patDemo.getZipCode() != null)
                {
                	zipCode = patDemo.getZipCode();
                }
                
                if (patDemo.getSSN() != null)
                {
                	fmpssn = patDemo.getSSN();
                }
            	
                if (patDemo.getPrimaryProvider() != null)
                {
                	pcm = patDemo.getPrimaryProvider();
                }
                
                if (patDemo.getPcmPhone() != null)
                {
                	pcmPhone = patDemo.getPcmPhone();
                }
                
                if (patDemo.getAge() != null)
                {
                	age = patDemo.getAge();
                }
                
                if (patDemo.getDob() != null)
                {
                	dob = patDemo.getDob();
                }
                
                if (patDemo.getGender() != null)
                {
                	gender = patDemo.getGender();
                }
                
                if (patDemo.getPhone1() != null)
                {
                	phone1 = patDemo.getPhone1();
                }
                
                if (patDemo.getPhone2() != null)
                {
                	phone2 = patDemo.getPhone2();
                }

                if (patDemo.getCodeGreen() != null)
                {
                	if (patDemo.getCodeGreen().equals("1"))
                	{
                		isCodeGray = true;
                        String msg = getServiceHandler().getPatientCodeGreen(createQueryBean(provider, patient, null, null, null, null, patDemo.getSiteCode(), 0));
                        if (msg != null)
                        {
                            codeGrayMsg = msg;
                        }
                	}
                    //pull code gray from non-cache VA sites
                    else if (!patDemo.getCodeGreen().equals("0") && patDemo.getCodeGreen().length() > 0)
                    {
                        isCodeGray = true;
                        codeGrayMsg = patDemo.getCodeGreen();
                    }
                }

                if (patDemo.getPercentServiceConnected() != null)
                {
                	percentServiceConnected = patDemo.getPercentServiceConnected();
                }
                
                if (patDemo.getClinic() != null)
                {
                	clinic = patDemo.getClinic();
                }
            	
            	GUIPatientDemographics guiPatDemo = new GUIPatientDemographics();
                guiPatDemo.setName(name);
                guiPatDemo.setAddress1(address1);
                guiPatDemo.setAddress2(address2);
                guiPatDemo.setCity(city);
                guiPatDemo.setState(state);
                guiPatDemo.setZipCode(zipCode);
                guiPatDemo.setFmpssn(fmpssn);
                guiPatDemo.setPcm(pcm);
                guiPatDemo.setPcmPhone(pcmPhone);
                guiPatDemo.setAge(age);
                guiPatDemo.setDob(dob);
                guiPatDemo.setGender(gender);
                guiPatDemo.setPhone1(phone1);
                guiPatDemo.setPhone2(phone2);
                guiPatDemo.setCodeGray(isCodeGray);
                guiPatDemo.setCodeGrayMessage(codeGrayMsg);
                guiPatDemo.setPercentServiceConnected(percentServiceConnected);
                guiPatDemo.setClinic(clinic);
                addSiteDataToGUIBean(guiPatDemo, patDemo);
                guiPatDemos.add(guiPatDemo);
            }
            
            //sort by site moniker
            Collections.sort(guiPatDemos, new SiteAlphaComparator());

            //push lookup site to the top of the list
            String providerLoginSiteCode = provider.getLoginSiteCode();

            int localSiteIdx = -1;
            for(int i = 0; i < guiPatDemos.size(); i++)
            {
                GUIPatientDemographics guiPatDemo = guiPatDemos.get(i);
                if (guiPatDemo.getSiteCode().equals(providerLoginSiteCode))
                {
                    localSiteIdx = i;
                    break;
                }
            }

            if (localSiteIdx > -1)
            {
                GUIPatientDemographics guiPatDemo = guiPatDemos.get(localSiteIdx);
                guiPatDemos.remove(localSiteIdx);
                guiPatDemos.add(0, guiPatDemo);
            }
            
            return guiPatDemos;
        } catch (SOAPException ex) {
            throw new DaoException(ex);
        }
    }
    
    public List<GUIPatientDemographicDetail> getPatientDemographicDetails(GUIProvider provider, GUIPatient patient)
    {
        try {
            List<PatientDemographicsDetail> patDemoDetails = getServiceHandler().getAllPatientDemographicDetails(createQueryBean(provider, patient));
            List<GUIPatientDemographicDetail> guiPatDemoDetails = new ArrayList<GUIPatientDemographicDetail>();

            for(PatientDemographicsDetail patDemoDetail : patDemoDetails)
            {
                GUIPatientDemographicDetail guiDemoDetail = new GUIPatientDemographicDetail();
                guiDemoDetail.setReport(patDemoDetail.getReport());

                addSiteDataToGUIBean(guiDemoDetail, patDemoDetail);

                guiPatDemoDetails.add(guiDemoDetail);
            }
            
            //sort by site moniker
            Collections.sort(guiPatDemoDetails, new SiteAlphaComparator());

            //push lookup site to the top of the list
            String providerLoginSiteCode = provider.getLoginSiteCode();

            int localSiteIdx = -1;
            for(int i = 0; i < guiPatDemoDetails.size(); i++)
            {
                GUIPatientDemographicDetail guiPatDemo = guiPatDemoDetails.get(i);
                if (guiPatDemo.getSiteCode().equals(providerLoginSiteCode))
                {
                    localSiteIdx = i;
                    break;
                }
            }

            if (localSiteIdx > -1)
            {
                GUIPatientDemographicDetail guiPatDemo = guiPatDemoDetails.get(localSiteIdx);
                guiPatDemoDetails.remove(localSiteIdx);
                guiPatDemoDetails.add(0, guiPatDemo);
            }
            
            return guiPatDemoDetails;
        } catch (SOAPException ex) {
            throw new DaoException(ex);
        }
    }
}
