/*
 * Janus 4.0 (c)
 * Copyright (c) 2011 Hawaii Resource Group LLC. All Rights Reserved.
 * Developed for the Pacific Telehealth & Technology Hui and the Pacific Joint Information Technology Center
 * Contributors:
 *             Honorable Senator Daniel K. Inouye
 *             VA Pacific Islands Health Care System
 *             Tripler Army Medical Center
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with the License.
 *
 * You may obtain a copy of the License at:
 *
 *            http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and limitations under the License.
 */

package gov.va.med.dao;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.UUID;

import gov.va.med.common.AppConstants;
import gov.va.med.guibeans.*;
import gov.va.med.vhahon.common.Utils;
import gov.va.med.jmeadows.webservice.Consult;

import gov.va.med.jmeadows.webservice.ProgressNote;

import gov.va.med.jmeadows.webservice.QueryBean;

/**
 * 
 *
 */
public class NoteDao extends DaoBase {
	public List<GUIProgressNote> getPatientDischargeSummaries(GUIProvider provider,
                                                              GUIPatient patient, Calendar startDate, Calendar endDate) {
		try {
			ProgressNote[] inpatientNotes = getServiceHandler()
					.getPatientDischargeSummaries(
							createQueryBean(provider, patient, startDate,
									endDate));

			List<GUIProgressNote> guiProgressNotes = new ArrayList<GUIProgressNote>();

			for (int i = 0; i < inpatientNotes.length; i++) {
				ProgressNote note = inpatientNotes[i];
				GUIProgressNote guiNote = mapToGUIProgressNote(note);

				if ("UNKNOWN".equalsIgnoreCase(guiNote.getNoteType())) {
					String noteType = guiNote.getLocation();
					guiNote.setLocation("");
					guiNote.setNoteType(noteType);
				}

				//notes need to be cached, unless data is from vista cache
                if (!AppConstants.PLATFORM_VISTA_CACHE.equals(guiNote.getSourcePlatform()))
                {
					guiNote.setSessionCachedRecord(true);

					// if note id is not null or empty, use note id
					if (note.getNoteId() != null
							&& note.getNoteId().length() > 0) {
						guiNote.setNoteId("cached:" + note.getNoteId() + "^" + i);
					}
					// else rely on list position
					else {
						guiNote.setNoteId("cached:^"+i);
					}

                    String format = guiNote.getStatus();

                    //if format is null, assume format is text
                    if (format != null &&
                            (AppConstants.PLATF0RM_SHARE.equalsIgnoreCase(note.getSourcePlatform()))
                            && !(format.contains("text") || format.contains("TEXT")
                                || format.contains("txt") || format.contains("TXT")))
                    {
                        guiNote.setNonPlainTextNote(true);
                    }
				}

				guiProgressNotes.add(guiNote);
			}

			return guiProgressNotes;

		} catch (Exception e) {
			throw new DaoException(e);
		}
	}

	public List<GUIProgressNote> getPatientProgressNotes(GUIProvider provider,
                                                         GUIPatient patient, Calendar startDate, Calendar endDate) {
		try {
			ProgressNote[] notes = getServiceHandler().getPatientProgressNotes(
					createQueryBean(provider, patient, startDate, endDate));
			List<GUIProgressNote> guiNotes = new ArrayList<GUIProgressNote>();

			for (int i = 0; i < notes.length; i++) {
				ProgressNote note = notes[i];
				GUIProgressNote guiNote = mapToGUIProgressNote(note);

				//notes need to be cached, unless data is from vista cache
				if (!AppConstants.PLATFORM_VISTA_CACHE.equals(guiNote.getSourcePlatform())) {
					guiNote.setSessionCachedRecord(true);

					// if note id is not null or empty, use note id
					if (note.getNoteId() != null
							&& note.getNoteId().length() > 0) {
						guiNote.setNoteId("cached:"
								+ note.getNoteId() + "^" + i);
					}
					// else rely on list position
					else {
						guiNote.setNoteId("cached:^" + i);
					}

                    if (AppConstants.PLATF0RM_EDR.equals(guiNote.getSourcePlatform()) ||
                        AppConstants.PLATF0RM_BHIE.equals(guiNote.getSourcePlatform()) ||
                            AppConstants.PLATF0RM_CHCS.equals(guiNote.getSourcePlatform()))
                    {
                        String format = guiNote.getStatus();

                        //if format is null, assume format is text
                        if (format != null && !(format.contains("text") || format.contains("TEXT")
                                || format.contains("txt") || format.contains("TXT")))
                        {
                            guiNote.setNonPlainTextNote(true);
                        }
                    }
				}

				guiNotes.add(guiNote);
			}

			return guiNotes;

		} catch (Exception e) {
			throw new DaoException(e);
		}
	}

	public GUIProgressNote getProgressNote(GUIProvider provider, GUIPatient patient,
			String noteId, String recordSiteCode, String recordSiteMoniker,
            String recordSiteAgency, String recordSourcePlatform) {
		try {
			String noteText = getServiceHandler().getProgressNote(
					createQueryBean(provider, patient, noteId, recordSiteCode));
			GUIProgressNote progressNote = new GUIProgressNote();
			progressNote.setNoteText(noteText);
			progressNote.setSiteCode(recordSiteCode);
            progressNote.setSiteMoniker(recordSiteMoniker);
            progressNote.setSiteAgency(recordSiteAgency);
            progressNote.setSourcePlatform(recordSourcePlatform);

			return progressNote;

		} catch (Exception e) {
			throw new DaoException(e);
		}
	}

	private GUIProgressNote mapToGUIProgressNote(ProgressNote note) {
		String noteDate = "";
		String location = "";
		String status = "";
		String noteType = "";
		String providerName = "";
		String visitAdm = "";
		String images = "";
		String amended = "";
		String noteId = "";
		String noteText = "";
        String patName = "";

		if (note.getNoteDate() != null) {
			noteDate = Utils.formatDisplayDate(note.getNoteDate());
		}

		if (note.getLocation() != null) {
			location = note.getLocation();
		}

		if (note.getStatus() != null) {
			status = note.getStatus();
		}

		if (note.getNoteType() != null) {
			noteType = note.getNoteType();
		}

		if (note.getProvider() != null) {
			providerName = note.getProvider();
		}

		if (note.getVisitDate() != null) {
			visitAdm = note.getVisitDate();
		}

		if (note.getImages() != null) {
			images = note.getImages();
		}

		if (note.getAmended() != null) {
			amended = note.getAmended();
		}

		if (note.getNoteId() != null) {
			noteId = note.getNoteId();
		}

		if (note.getNoteText() != null) {
			noteText = note.getNoteText();
		}

        if (note.getPatientName() != null) {
			patName = note.getPatientName();
		}

		GUIProgressNote guiNote = new GUIProgressNote();

		guiNote.setAmended(amended);
		guiNote.setNoteDate(noteDate);
		guiNote.setImages(images);
		guiNote.setLocation(location);
		guiNote.setNoteId(noteId);
		guiNote.setNoteType(noteType);
		guiNote.setProviderName(providerName);
		guiNote.setStatus(status);
		guiNote.setVisitAdm(visitAdm);

        addSiteDataToGUIBean(guiNote, note);

        guiNote.setNoteText(noteText);
        guiNote.setPatientName(patName);

		return guiNote;
	}

	public List<GUIConsult> getPatientConsults(GUIProvider provider,
			GUIPatient patient, Calendar startDate, Calendar endDate) {
		try {
			Consult[] consults = getServiceHandler().getPatientConsultRequests(
					createQueryBean(provider, patient, startDate, endDate));

			List<GUIConsult> guiConsults = new ArrayList<GUIConsult>();

			for (int i = 0; i < consults.length; i++) {
				Consult consult = consults[i];

				String requestDate = "";
				String consultId = "";
				String outCode = "";
				String outDescription = "";
				String outStatus = "";
				String consultReport = "";
                String service = "";

				if (consult.getRequestDate() != null) {
					requestDate = Utils.formatDisplayDate(consult
							.getRequestDate());
				}

				if (consult.getService() != null) {
					outCode = consult.getService();
				}

				if (consult.getReport() != null) {
					consultReport = consult.getReport();
				}

				if (consult.getProcedureConsult() != null) {
					outDescription = consult.getProcedureConsult();
				}

                if (consult.getService() != null)
                {
                    service = consult.getService();
                }

				if (consult.getStatus() != null)
                {
                    outStatus = consult.getStatus();
                }

				String newConsultId = "";
				if (consult.getId() != null) {
					newConsultId = consult.getId();
				}

				String authNum = Utils.getPiece(consult.getId(), "^", 2);
				if (authNum != null && authNum.length() > 0) {
					outCode = outCode + "  #" + authNum;
				}

				// if va record and previous consult id equals current
				// then mark as duplicate record
				boolean isDuplicate = newConsultId.equals(consultId)
						&& (AppConstants.PLATFORM_VISTA_CACHE.equals(consult.getSourcePlatform()) ||
                        AppConstants.PLATFORM_VISTA_RPC.equals(consult.getSourcePlatform()));

				// skip duplicates
				if (!isDuplicate) {
					GUIConsult guiConsult = new GUIConsult();
					//if CHCS platform, mark as cached
                    if (AppConstants.PLATF0RM_CHCS.equals(consult.getSourcePlatform())) {
						guiConsult.setSessionCachedRecord(true);
						consultId = "cached:^" + i;
					} else {
						consultId = newConsultId;
					}

					guiConsult.setConsultId(consultId);
					guiConsult.setOutCode(outCode);
					guiConsult.setOutDescription(outDescription);
					guiConsult.setOutStatus(outStatus);
					guiConsult.setRequestDate(requestDate);
					guiConsult.setReport(consultReport);
                    guiConsult.setService(service);

                    if (consultReport.contains("CISURL"))
                    {
                        guiConsult.setNonPlainTextNote(true);
                    }

					addSiteDataToGUIBean(guiConsult, consult);

					guiConsults.add(guiConsult);
				}
			}

			return guiConsults;

		} catch (Exception ex) {
			throw new DaoException(ex);
		}
	}

	public GUIConsult getConsultReport(GUIProvider provider, GUIPatient patient,
			String recordId, String recordSiteCode,
            String recordSiteMoniker, String recordSiteAgency,
            String recordSourcePlatform) {
		try {
			String[] report = getServiceHandler()
					.getConsultReport(
							createQueryBean(provider, patient, recordId,
									recordSiteCode));
			GUIConsult consult = new GUIConsult();

			if (report != null) {

				String reportResult = "";
                int i = 0;
				for (String line : report) {
                    if (line!=null){
                        line.replaceAll("<", "&lt;");
                        line.replaceAll(">", "&gt;");
                    }
                    else {
                        line = "";
                    }
					reportResult += line + "\n";
				}

				consult.setReport(reportResult);
				consult.setSiteCode(recordSiteCode);
                consult.setSiteMoniker(recordSiteMoniker);
                consult.setSiteAgency(recordSiteAgency);
                consult.setSourcePlatform(recordSourcePlatform);
			}

			return consult;

		} catch (Exception e) {
            e.printStackTrace(System.out);
			throw new DaoException(e);
		}
	}

	public GUIEssentrisNote getEssentrisNote(GUIProvider provider,
			GUIPatient patient, String noteId, String noteFormat) {
		try {
			String noteText = getServiceHandler().getCISNote(
					createQueryBean(provider, patient, null, null, noteFormat,
							noteId, "", 0));

			GUIEssentrisNote guiEssentrisNote = new GUIEssentrisNote();
			guiEssentrisNote.setNoteText(noteText);
			guiEssentrisNote.setSiteAgency(AppConstants.AGENCY_DOD);

			return guiEssentrisNote;
		} catch (Exception e) {
			throw new DaoException(e);
		}
	}

    public List<GUIConsult> getProviderConsultsRequested(
            GUIProvider provider,
			Calendar startDate,
            Calendar endDate,
            byte[] encryptSecurityKey) {
		try {
			Consult[] consults = getServiceHandler().getProviderConsultsRequested(
					createQueryBean(provider, null, startDate, endDate));

			List<GUIConsult> guiConsults = new ArrayList<GUIConsult>();

			for (int i = 0; i < consults.length; i++) {
				Consult consult = consults[i];

				String requestDate = "";
				String consultId = "";
				String outCode = "";
				String outDescription = "";
				String outStatus = "";
				String consultReport = "";
                String service = "";
                String patName = "";
                String patToken = "";

				if (consult.getRequestDate() != null) {
					requestDate = Utils.formatDisplayDate(consult
							.getRequestDate());
				}

				if (consult.getService() != null) {
					outCode = consult.getService();
				}

    			if (consult.getReport() != null) {
					consultReport = consult.getReport();
				}

				if (consult.getProcedureConsult() != null) {
					outDescription = consult.getProcedureConsult();
				}

                if (consult.getService() != null)
                {
                    service = consult.getService();
                }

				if (consult.getStatus() != null)
                {
                    outStatus = consult.getStatus();
                }

				String newConsultId = "";
				if (consult.getId() != null) {
					newConsultId = consult.getId();
				}

				String authNum = Utils.getPiece(consult.getId(), "^", 2);
				if (authNum != null && authNum.length() > 0) {
					outCode = outCode + "  #" + authNum;
				}

                if (consult.getPatientName() != null){
                    patName = consult.getPatientName();
                }

                patToken = generatePatientToken(encryptSecurityKey, consult.getPatientId(), consult.getSiteCode());


				// if vista record and previous consult id equals current
				// then mark as duplicate record
				boolean isDuplicate = newConsultId.equals(consultId)
						&& (AppConstants.PLATFORM_VISTA_CACHE.equals(consult.getSourcePlatform()) ||
                                AppConstants.PLATFORM_VISTA_RPC.equals(consult.getSourcePlatform()));

				// skip duplicates
				if (!isDuplicate) {
					GUIConsult guiConsult = new GUIConsult();
					if (AppConstants.PLATF0RM_CHCS.equals(consult.getSourcePlatform()))
                    {
						guiConsult.setSessionCachedRecord(true);
						consultId = "cached:^" + i;
					} else {
						consultId = newConsultId;
					}

					guiConsult.setConsultId(consultId);
					guiConsult.setOutCode(outCode);
					guiConsult.setOutDescription(outDescription);
					guiConsult.setOutStatus(outStatus);
					guiConsult.setRequestDate(requestDate);
					guiConsult.setReport(consultReport);
                    guiConsult.setService(service);

                    if (consultReport.contains("CISURL"))
                    {
                        guiConsult.setNonPlainTextNote(true);
                    }

					addSiteDataToGUIBean(guiConsult, consult);

                    guiConsult.setPatientName(patName);
                    guiConsult.setPatientToken(patToken);

					guiConsults.add(guiConsult);
				}
			}

			return guiConsults;

		} catch (Exception ex) {
			throw new DaoException(ex);
		}
	}

     public List<GUIConsult> getProviderConsultsReceived(
            GUIProvider provider,
			Calendar startDate,
            Calendar endDate,
            byte[] encryptSecurityKey) {
		try {
			Consult[] consults = getServiceHandler().getProviderConsultsReceived(
					createQueryBean(provider, null, startDate, endDate));

			List<GUIConsult> guiConsults = new ArrayList<GUIConsult>();

			for (int i = 0; i < consults.length; i++) {
				Consult consult = consults[i];

				String requestDate = "";
				String consultId = "";
				String outCode = "";
				String outDescription = "";
				String outStatus = "";
				String consultReport = "";
                String service = "";
                String patName = "";
                String patToken = "";

				if (consult.getRequestDate() != null) {
					requestDate = Utils.formatDisplayDate(consult
							.getRequestDate());
				}

				if (consult.getService() != null) {
					outCode = consult.getService();
				}

				if (consult.getReport() != null) {
					consultReport = consult.getReport();
				}

				if (consult.getProcedureConsult() != null) {
					outDescription = consult.getProcedureConsult();
				}

                if (consult.getService() != null)
                {
                    service = consult.getService();
                }

				if (consult.getStatus() != null)
                {
                    outStatus = consult.getStatus();
                }

				String newConsultId = "";
				if (consult.getId() != null) {
					newConsultId = consult.getId();
				}

				String authNum = Utils.getPiece(consult.getId(), "^", 2);
				if (authNum != null && authNum.length() > 0) {
					outCode = outCode + "  #" + authNum;
				}

                if (consult.getPatientName()!=null){
                    patName = consult.getPatientName();
                }

                patToken = generatePatientToken(encryptSecurityKey, consult.getPatientId(), consult.getSiteCode());

				// if vista record and previous consult id equals current
				// then mark as duplicate record
				boolean isDuplicate = newConsultId.equals(consultId)
						&& (AppConstants.PLATFORM_VISTA_CACHE.equals(consult.getSourcePlatform()) ||
                                AppConstants.PLATFORM_VISTA_RPC.equals(consult.getSourcePlatform()));

				// skip duplicates
				if (!isDuplicate) {
					GUIConsult guiConsult = new GUIConsult();
					if (AppConstants.PLATF0RM_CHCS.equals(consult.getSourcePlatform()))
                    {
						guiConsult.setSessionCachedRecord(true);
						consultId = "cached:^" + i;
					} else {
						consultId = newConsultId;
					}

					guiConsult.setConsultId(consultId);
					guiConsult.setOutCode(outCode);
					guiConsult.setOutDescription(outDescription);
					guiConsult.setOutStatus(outStatus);
                    guiConsult.setRequestDate(requestDate);
					guiConsult.setReport(consultReport);
                    guiConsult.setService(service);

                    if (consultReport.contains("CISURL"))
                    {
                        guiConsult.setNonPlainTextNote(true);
                    }

					addSiteDataToGUIBean(guiConsult, consult);

                    guiConsult.setPatientName(patName);
                    guiConsult.setPatientToken(patToken);

					guiConsults.add(guiConsult);
				}
			}

			return guiConsults;

		} catch (Exception ex) {
			throw new DaoException(ex);
		}
	}

    public List<GUIProgressNote> getProviderUnsignedNotes(
            GUIProvider provider,
			Calendar startDate,
            Calendar endDate,
            byte[] encryptSecurityKey) {
		try {
			ProgressNote[] notes = getServiceHandler().getProviderUnsignedNotes(
					createQueryBean(provider, null, startDate, endDate));
			List<GUIProgressNote> guiNotes = new ArrayList<GUIProgressNote>();

			for (int i = 0; i < notes.length; i++) {
				ProgressNote note = notes[i];
				GUIProgressNote guiNote = mapToGUIProgressNote(note);
                guiNote.setPatientToken(generatePatientToken(encryptSecurityKey, note.getPatientId(), note.getSiteCode()));

				// all non-vista cache notes are delivered with initial call
				// need to be cached
                if (!AppConstants.PLATFORM_VISTA_CACHE.equals(guiNote.getSourcePlatform()))
                {
					guiNote.setSessionCachedRecord(true);

					// if note id is not null or empty, use note id
					// list position
					if (note.getNoteId() != null
							&& note.getNoteId().length() > 0) {
						guiNote.setNoteId("cached:"
								+ note.getNoteId() + "^" + i);
					}
					// else rely on list position
					else {
						guiNote.setNoteId("cached:^" + i);
					}
                    //if CHCS record
                    if (AppConstants.PLATF0RM_CHCS.equals(guiNote.getSourcePlatform()))
                    {
                        String format = guiNote.getStatus();

                        //if format is null, assume format is text
                        if (format != null && !(format.equalsIgnoreCase("text") || format.equalsIgnoreCase("txt")))
                        {
                            guiNote.setNonPlainTextNote(true);
                        }
                    }
				}


				guiNotes.add(guiNote);
			}

			return guiNotes;

		} catch (Exception e) {
			throw new DaoException(e);
		}
	}

    public String addNote(
            GUIProvider provider,
            GUIPatient patient,
            ProgressNote progressNote,
            String sig) {

        String returnVal = "";
		try {
            QueryBean qb =  createQueryBean(provider, patient);
            qb.setItemId(sig);
			returnVal = getServiceHandler().createNewSignedNote(
					qb,
                    progressNote);

		} catch (Exception ex) {
			//throw new DaoException(ex);
            returnVal = "Error: " + ex.getMessage();
		}

        return returnVal;
	}
}
