/*
 * Janus 4.0 (c)
 * Copyright (c) 2011 Hawaii Resource Group LLC. All Rights Reserved.
 * Developed for the Pacific Telehealth & Technology Hui and the Pacific Joint Information Technology Center
 * Contributors:
 *             Honorable Senator Daniel K. Inouye
 *             VA Pacific Islands Health Care System
 *             Tripler Army Medical Center
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with the License.
 *
 * You may obtain a copy of the License at:
 *
 *            http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and limitations under the License.
 */

package gov.va.med.dao;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import gov.va.med.common.AppConstants;
import gov.va.med.guibeans.*;
import gov.va.med.vhahon.common.Utils;
import gov.va.med.jmeadows.webservice.LabOrder;
import gov.va.med.jmeadows.webservice.LabResult;



import javax.xml.datatype.XMLGregorianCalendar;

public class LabDao extends DaoBase
{
	public List<GUILabOrder> getPatientLabs(GUIProvider provider, GUIPatient patient, Calendar startDate, Calendar endDate)
	{
		try {
			LabOrder[] labs = getServiceHandler().getPatientLabs(createQueryBean(provider, patient, startDate, endDate));
			
			List<GUILabOrder> guiLabs = new ArrayList<GUILabOrder>();
			
			for(LabOrder lab : labs)
			{
				String testName = "";
				String collectionDate = "";
				String sample = "";
				String chemId = "";
				String chemORT = "";
				String subscript = "";
				
				if (lab.getTestName() != null)
				{
					testName = lab.getTestName();
				}
				
				if (lab.getCollectionDate() != null)
				{
					collectionDate = Utils.formatDisplayDate(lab.getCollectionDate());
				}
				
				if (lab.getCollectionSample() != null)
				{
					sample = lab.getCollectionSample();
				}
				
				if (lab.getOrderId() != null)
				{
					chemId = lab.getOrderId();
				}
				
				if (lab.getInternalId() != null)
				{
					chemORT = lab.getInternalId();
				}
				
				if (lab.getSubscript() != null)
				{
					subscript = lab.getSubscript();
				}
				
				GUILabOrder guiLab = new GUILabOrder();

                //CH results contain a panel list
                if (subscript.equals("CH") || subscript.contains("CHEM") || subscript.contains("chem"))
                {
                    guiLab.setNonPlainTextResult(true);
                }

                guiLab.setTestName(testName);
				guiLab.setCollectionDate(collectionDate);
				guiLab.setSample(sample);
				guiLab.setChemId(chemId);
				guiLab.setChemORT(chemORT);
				guiLab.setSubscript(subscript);
				addSiteDataToGUIBean(guiLab, lab);
				
				guiLabs.add(guiLab);
			}
			
			return guiLabs;
			
		} catch (Exception e) {
			throw new DaoException(e);
		}		
	}

	public List<GUILabResult> getLabOrderResults(GUIProvider provider, GUIPatient patient, String chemORT, String subscript,
                                                 String recordSiteCode)
	{
		try {
			LabResult[] labResults = getServiceHandler().getLabOrderResult(createQueryBean(provider, patient, null, null, subscript, chemORT, recordSiteCode, 0));
			
			List<GUILabResult> guiLabResults = new ArrayList<GUILabResult>();
			
			for(LabResult labResult : labResults)
			{
               guiLabResults.add(toGUILabResult(labResult));
			}
			
			return guiLabResults;
			
		} catch (Exception e) {
			throw new DaoException(e);
		}
	}

    private GUILabResult toGUILabResult(LabResult labResult)
    {
        String testId = "";
        String result = "";
        String accession = "";
        String orderId = "";
        String testName = "";
        String printName = "";
        String verifiedBy = "";
        String hiLoFlag = "";
        String referenceRange = "";
        String resultStatus = "";
        String units = "";
        String comment = "";
        String resultDate = "";
        String specimen = "";
        String patName = "";
        String patToken = "";

        if (labResult.getTestId() != null)
        {
            testId = labResult.getTestId();
        }

        if (labResult.getResult() != null)
        {
            result = labResult.getResult();
        }

        if (labResult.getOrderId() != null)
        {
            orderId = labResult.getOrderId();
        }

        if (labResult.getTestName() != null)
        {
            testName = labResult.getTestName();
        }

        if (labResult.getPrintName() != null)
        {
            printName = labResult.getPrintName();
        }

        if (labResult.getVerifiedBy() != null)
        {
            verifiedBy = labResult.getVerifiedBy();
        }

        if (labResult.getHiLoFlag() != null)
        {
            hiLoFlag = labResult.getHiLoFlag();
        }

        if (labResult.getReferenceRange() != null)
        {
            referenceRange = labResult.getReferenceRange();
        }

        if (labResult.getResultStatus() != null)
        {
            resultStatus = labResult.getResultStatus();
        }

        if (labResult.getUnits() != null)
        {
            units = labResult.getUnits();
        }

        if (labResult.getComment() != null)
        {
            comment = labResult.getComment();
        }

        if (labResult.getResultDate() != null)
        {
            resultDate = Utils.formatDisplayDate(labResult.getResultDate());
        }

        if (labResult.getSpecimen() != null)
        {
            specimen = labResult.getSpecimen();
        }

        if (labResult.getAccession() != null)
        {
            accession = labResult.getAccession();
        }

        if (labResult.getPatientName()!=null){
            patName = labResult.getPatientName();
        }

        GUILabResult guiLabResult = new GUILabResult();

        //graphing only available on Vista cache and CHCS platforms
        if (AppConstants.PLATFORM_VISTA_CACHE.equals(labResult.getSourcePlatform()) ||
                AppConstants.PLATF0RM_CHCS.equals(labResult.getSourcePlatform()))
        {
            try
            {
                //only allow histogram for numeric results
                Double.parseDouble(result);
                guiLabResult.setHistogramAvailable(true);
            }
            catch (Exception e)
            {

            }
        }
        else guiLabResult.setHistogramAvailable(false);

        guiLabResult.setTestId(testId);
        guiLabResult.setSpecimen(specimen);
        guiLabResult.setResult(result);
        guiLabResult.setAccession(accession);
        guiLabResult.setOrderId(orderId);
        guiLabResult.setTestName(testName);
        guiLabResult.setPrintName(printName);
        guiLabResult.setVerifiedBy(verifiedBy);
        guiLabResult.setHiLoFlag(hiLoFlag);
        guiLabResult.setReferenceRange(referenceRange);
        guiLabResult.setUnits(units);
        guiLabResult.setResultStatus(resultStatus);
        guiLabResult.setComment(comment);
        guiLabResult.setResultDate(resultDate);
        guiLabResult.setSpecimen(specimen);
        addSiteDataToGUIBean(guiLabResult, labResult);
        guiLabResult.setPatientName(patName);

        return guiLabResult;
    }

    public GUILabGraph getLabGraphData(GUIProvider provider, GUIPatient patient, String itemId, Calendar startDate, Calendar endDate, String recordSiteCode)
    {
        try {
            LabResult[] labResults = getServiceHandler().getPatientLabTestResults(
                    createQueryBean(provider, patient, startDate, endDate, itemId, recordSiteCode));

            return toGUILabGraph(labResults);

        } catch (Exception e) {
            throw new DaoException(e);
        }
    }

    private GUILabGraph toGUILabGraph(LabResult[] labResults)
    {
        XMLGregorianCalendar date;
        double result = 0;
        XMLGregorianCalendar vD;
        String vR = "";

        int count = 0;
        double total = 0;
        double min = 0;
        double max = 0;
        double avg = 0;
        double hFactor = 0;
        XMLGregorianCalendar mindate = null;
        XMLGregorianCalendar maxdate = null;

        List<GUILabResult> guiLabResults = new ArrayList<GUILabResult>();
        GUILabGraph guiLabGraph = new GUILabGraph();

        for (LabResult res : labResults)
        {
            try
            {
                vD = res.getResultDate();
                vR = res.getResult();

                if (vD == null || vR == null) continue;

                 result = new Double(vR).doubleValue();
                 guiLabResults.add(toGUILabResult(res));
            }
            catch (Exception e)
            {
                //do not include non-numeric results in lab graph result set
                guiLabGraph.setExceptionFlag(true);
                Utils.logMessage("Bad value in lab result cell: '" + vR + "'");
                continue;
            }
            date = vD;
            count++;
            total = total + result;
            if (count == 1) {
                min = result;
                max = result;
                mindate = date;
                maxdate = date;
            }
            else {
                if (min > result) {
                    min = result;
                    mindate = date;
                }
                if (max < result) {
                    max = result;
                    maxdate = date;
                }
            }
        }

        guiLabGraph.setGuiLabResults(guiLabResults);

		if (count > 0) avg = total/count;

        guiLabGraph.setCount(count);
        guiLabGraph.setAvg(roundAverage(avg));
        guiLabGraph.setTotal(total);
        guiLabGraph.setMax(max);
        guiLabGraph.setMin(min);
        guiLabGraph.setMinDate(Utils.formatDisplayDate(mindate));
        guiLabGraph.setMaxDate(Utils.formatDisplayDate(maxdate));

        return guiLabGraph;
    }

    private double roundAverage(Double avg) {
        String s = new Double(avg).toString();
        int i = s.indexOf('.');
        if (i >= 0) {
            int o = 0;
            if (s.length() > i+3) {
                o = new Integer(s.substring(i+3,i+4)).intValue();
                s = s.substring(0,i+3);
            }
            avg = new Double(s).doubleValue();
            if (o > 4) avg += .01;
        }
        return avg;
    }

    public List<GUILabResult> getProviderLabAbnormalResults(
            GUIProvider provider,
            Calendar startDate,
            Calendar endDate,
            byte[] encryptSecurityKey)
	{
		try {
			LabResult[] labResults = getServiceHandler().getProviderLabAbnormalResults(createQueryBean(provider, null, startDate, endDate));

			List<GUILabResult> guiLabResults = new ArrayList<GUILabResult>();

			for(LabResult labResult : labResults)
			{
                GUILabResult lr = toGUILabResult(labResult);
                lr.setPatientToken(generatePatientToken(encryptSecurityKey, labResult.getPatientId(), labResult.getSiteCode()));
                guiLabResults.add(lr);
			}

			return guiLabResults;

		} catch (Exception e) {
			throw new DaoException(e);
		}
	}
}
