/*
 * Janus 4.0 (c)
 * Copyright (c) 2011 Hawaii Resource Group LLC. All Rights Reserved.
 * Developed for the Pacific Telehealth & Technology Hui and the Pacific Joint Information Technology Center
 * Contributors:
 *             Honorable Senator Daniel K. Inouye
 *             VA Pacific Islands Health Care System
 *             Tripler Army Medical Center
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with the License.
 *
 * You may obtain a copy of the License at:
 *
 *            http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and limitations under the License.
 */

package janusgrails

import grails.converters.*
import gov.va.med.common.AESCryptoUtil
import gov.va.med.common.HttpKeys
import gov.va.med.vhahon.common.Utils

class BaseController {

    protected sendResponse(data)
    {
        withFormat
        {
            json {render data as JSON}
            xml {render data as XML}
        }
    }

    /**
     * Date Range Helper method will provide controller with valid start and end dates if
     * they were not passed in by request. Start and end dates are placed into the request object.
     */
    protected void dateRangeCheck()
    {
        String startDateParam = params[HttpKeys.START_DATE];
        String endDateParam = params[HttpKeys.END_DATE];

        //if no start date param, set default to T-3 months
        if (startDateParam == null || startDateParam.length() < 2)
        {
            Calendar startCal = Calendar.getInstance();
            startCal.add(Calendar.MONTH, -3);
            startDateParam = Utils.formatDate(startCal.getTime());
        }

        //if no end date param, set default to today's date
        if (endDateParam == null || endDateParam.length() < 2)
        {
            Calendar endCal = Calendar.getInstance();
            endDateParam = Utils.formatDate(endCal.getTime());
        }

        //push start and end date params into request object for controller use
        request[HttpKeys.START_DATE] = startDateParam;
        request[HttpKeys.END_DATE] = endDateParam;
    }
    
    protected generateRecordDetailsToken(String patientId, String patientSiteCode,
                                         String recordControllerName, Map recordQuery,
                                         String recordSiteCode, String recordSiteMoniker,
                                         String recordSiteAgency, String recordSourcePlatform)
    {
        return generateRecordDetailsToken(patientId, patientSiteCode,
                recordControllerName, recordQuery, recordSiteCode, recordSiteMoniker,
                recordSiteAgency, recordSourcePlatform, false)
    }

    protected generateRecordDetailsTokenWithPatToken(String patToken, String recordControllerName,
                                                     Map recordQuery,
                                                     String recordSiteCode, String recordSiteMoniker, String recordSiteAgency,
                                                     String recordSourcePlatform, boolean isSessionCached)
    {
        Map patientParamsMap = AESCryptoUtil.decryptQuery(session.appUser.securityKey,patToken);
        String patId = patientParamsMap[HttpKeys.PATIENT_ID];
        String patSiteCode = patientParamsMap[HttpKeys.PATIENT_SITE_CODE]
        return generateRecordDetailsToken(patId, patSiteCode, recordControllerName,
                recordQuery, recordSiteCode, recordSiteMoniker, recordSiteAgency, recordSourcePlatform, isSessionCached);
    }
    
    protected generateRecordDetailsToken(String patientId, String patientSiteCode, String recordControllerName,
                            Map recordQuery, String recordSiteCode, String recordSiteMoniker, String recordSiteAgency,
                            String recordSourcePlatform, boolean isSessionCached)
    {
        if (recordQuery == null)
        {
            throw new RuntimeException("record details token requires a record query map")
        }
        
        def recordQueryStr = ""
        
        Object[] keys = recordQuery.keySet().toArray()
        
        for(Object key : keys)
        {
            recordQueryStr += "${key}=${recordQuery.get(key)}&"
        }
        
        recordQueryStr += 
                        "${HttpKeys.PATIENT_ID}=${patientId}&"+
                        "${HttpKeys.PATIENT_SITE_CODE}=${patientSiteCode}&"+
                        "${HttpKeys.RECORD_CONTROLLER_NAME}=${recordControllerName}&"+
                        "${HttpKeys.RECORD_SITE_CODE}=${recordSiteCode}&"+
                        "${HttpKeys.RECORD_SITE_MONIKER}=${recordSiteMoniker}&"+
                        "${HttpKeys.RECORD_SITE_AGENCY}=${recordSiteAgency}&"+
                        "${HttpKeys.RECORD_SOURCE_PLATFORM}=${recordSourcePlatform}&"+
                        "${HttpKeys.SESSION_CACHED_RECORD}=${isSessionCached}&"+                        
                        "${HttpKeys.TIMESTAMP}=${System.currentTimeMillis()}"
        
        if (isSessionCached)
        {
            recordQueryStr += "&${HttpKeys.RECORD_CACHE_KEY}=${patientId}^${patientSiteCode}^${session.appUser.cacheCounter}";
            session.appUser.cacheCounter++
        }

        return AESCryptoUtil.generateEncryptedQuery(session.appUser.securityKey, recordQueryStr)
    }
    
    protected getCachedRecordKey(recordDetailsToken)
    {
        Map recordParamsMap = AESCryptoUtil.decryptQuery(session.appUser.securityKey, recordDetailsToken)

        return recordParamsMap.get(HttpKeys.RECORD_CACHE_KEY)
    }
}
