/*
 * Janus 4.0 (c)
 * Copyright (c) 2011 Hawaii Resource Group LLC. All Rights Reserved.
 * Developed for the Pacific Telehealth & Technology Hui and the Pacific Joint Information Technology Center
 * Contributors:
 *             Honorable Senator Daniel K. Inouye
 *             VA Pacific Islands Health Care System
 *             Tripler Army Medical Center
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with the License.
 *
 * You may obtain a copy of the License at:
 *
 *            http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and limitations under the License.
 */

import gov.va.med.common.HttpKeys
import gov.va.med.vhahon.common.Utils
import gov.va.med.dao.PatientDao
import gov.va.med.common.AESCryptoUtil
import gov.va.med.jmeadows.webservice.Patient

/**
 *
 *
 */
class SecurityFilters
{
    def filters = {

        /**
         * If invalid session, send user to authenticate controller's login action
         * If rest action, send 403 error
         */
        sessionCheck(controller:'*', action:'*')
        {
            before = {
                
                if(!session.appUser && controllerName != 'authenticate')
                {
                    //if rest action, send 403 error
                    if (actionName == "show" || actionName == "save" ||
                            actionName == "update" || actionName == "delete")
                    {
                        response.sendError 403;
                        return false;
                    }

                    //otherwise redirect to login screen
                    redirect(controller:'authenticate', action:'login')
                    return false
                }
            }
        }
                
        /**
         * Do not allow browser caching for all controllers/actions
         */
        noCacheCheck(controller:'*', action:'*')
        {
            before = {                
                response.setHeader('Cache-Control', 'no-cache, no-store')
                response.setDateHeader('Expires', (new Date()-1).time )
                response.setHeader('Pragma', 'no-cache')
            }
        }
        
        /**
         * If login action and a valid session exists, send
         * user to the provider page
         */
        loginRedirect(controller:"authenticate", action:"login")
        {
            before = {
                if (session.appUser)
                {
                    redirect(controller:'app')
                    return false
                }
            }
        }
        
        /**
         * Filter parses and verifies patient and record security tokens.
         * 
         */
        securityTokenCheck(controller:'*', action:'*')
        {
            before = {
                
               /** 
                * If valid patient token exists in request params,
                * patient tied to the token will be placed in the request object (request.patient).
                * 
                * A patient token is an AES encrypted base 64 string that consists
                * of four parameters: patient id, patient lookup site (site associated with the patient id),
                * user session id, and timestamp.
                */
                if (params[HttpKeys.PATIENT_TOKEN] && controllerName != "patientSelect")
                {
                    try
                    {
                        def patientToken = params[HttpKeys.PATIENT_TOKEN]
                        def provider = session.appUser.provider
                        def selectedPatientMap = session.appUser.selectedPatientMap
                        
                        if (patientToken == null || patientToken.length() < 1)
                        {
                            log.error("securityTokenCheck::Patient token null or empty")
                            throw new RuntimeException("Invalid patient token")
                        }
                        else if (provider == null || selectedPatientMap == null)
                        {
                            log.error("securityTokenCheck::Provider or selected patient map is null")
                            throw new RuntimeException("Uninitialized session")
                        }
                        
                        
                        //first check to see if patient is already mapped 
                        def selectedPatient = selectedPatientMap.get(patientToken)
            
                        if (selectedPatient == null)
                        {
                            def patientParams = AESCryptoUtil.decryptQuery(session.appUser.securityKey,patientToken)

                            def paramPatientSiteCode = patientParams[HttpKeys.PATIENT_SITE_CODE]
                            def paramPatientId = patientParams[HttpKeys.PATIENT_ID]
            
                            if(selectedPatientMap.get(patientToken) == null)
                            {
                                PatientDao patientDao = new PatientDao()
                                selectedPatient = patientDao.selectPatient(provider, paramPatientId, paramPatientSiteCode)
                            }

                            if (selectedPatient == null)
                            {
                                log.error("securityTokenCheck::selected patient is null")
                                throw new RuntimeException("Invalid patient token")
                            }

                            selectedPatientMap.put(patientToken, selectedPatient)
                        }

                        request.patient = selectedPatient
                    }
                    catch(Exception e)
                    {
                        response.sendError 500
                        return false;
                    }
                }
                /**
                 * If a valid record details exists in request params,
                 * necessary record variables (record query, isSessionCachedRecord)
                 * will be added to the request object.  
                 * 
                 * A record details token is tied to the user session.
                 */
                if (params[HttpKeys.RECORD_DETAILS_TOKEN])
                {
                    try
                    {
                        def recordDetailsToken = params[HttpKeys.RECORD_DETAILS_TOKEN]                       
                        def provider = session.appUser.provider
                        
                        if (recordDetailsToken == null || recordDetailsToken.length() < 1)
                        {
                            log.error("securityTokenCheck::Record details token is null or empty")
                            throw new RuntimeException("Invalid record details token")
                        }
                        else if (provider == null)
                        {
                            log.error("securityTokenCheck::Provider is null")
                            throw new RuntimeException("Uninitialized session")
                        }

                        def detailsParams = AESCryptoUtil.decryptQuery(session.appUser.securityKey, recordDetailsToken)

                        def paramPatientId = detailsParams[HttpKeys.PATIENT_ID]
                        def paramRecordControllerName = detailsParams[HttpKeys.RECORD_CONTROLLER_NAME]
                        def paramRecordSiteCode = detailsParams[HttpKeys.RECORD_SITE_CODE]
                        def paramSessionCachedRecord = detailsParams[HttpKeys.SESSION_CACHED_RECORD]

                        //record token security checks
                        if (request.patient.getPatientId() != paramPatientId ||
                            controllerName != paramRecordControllerName)
                        {
                             log.error("securityTokenCheck::record details token patient id, session id, or controller name is invalid.")
                             throw new RuntimeException("Invalid record details token")
                        }
           
                        request.recordQuery = detailsParams

                        request.isSessionCachedRecord = false
                        
                        if (paramSessionCachedRecord.equalsIgnoreCase("true"))
                        {
                            request.isSessionCachedRecord = true
                        }
                    }
                    catch(Exception e)
                    {
                        response.sendError 500
                        return false;
                    }
                }
            }
        }
    }   
}

